package diskmanager

import (
	"context"
	"time"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	pb "a.yandex-team.ru/infra/rsm/diskmanager/api"
	_ "a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	_ "a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/disk"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/lvm"
)

type Condition struct {
	Val bool
	Msg string
	TS  time.Time
}

type Status struct {
	Configured Condition
	Ready      Condition
	Absent     Condition
	Error      Condition
	Manageable Condition
}

type Disk struct {
	ID     string
	Status Status
	Final  *disk.Disk
	Parts  []*disk.Disk
	Pv     *lvm.PhysicalVolume
	Vg     *lvm.VolumeGroup
	Lvs    []*lvm.LogicalVolume
}

func (c *Condition) Set(val bool, msg string) {
	if val == c.Val && c.Msg == msg {
		return
	}
	c.Val = val
	c.Msg = msg
	c.TS = time.Now()
}

//func NewDisk()
func (d *Disk) MayModify() error {
	if d.Status.Error.Val {
		return status.Error(codes.FailedPrecondition, "disk is in error state")
	}
	if d.Status.Absent.Val {
		return status.Error(codes.FailedPrecondition, "disk is absent")
	}
	return nil

}

func validateFormatDiskRequest(req *pb.FormatDiskRequest) error {
	var end int64 = -1
	hasPv := false
	for _, p := range req.Partitions {
		if end > p.StartBytes {
			return status.Error(codes.InvalidArgument, "Partitions overlaps")
		}
		end = p.StartBytes + p.SizeBytes

		if p.VolumeSource {
			if hasPv {
				return status.Error(codes.InvalidArgument, "Multiple volumesources per disk")
			}
			hasPv = true
		}
		if p.Type == string(disk.PartTypeLvm) && !p.VolumeSource {
			return status.Error(codes.InvalidArgument, "request lvm partition type, but not volume source")
		}
	}
	return nil
}

func (d *Disk) format(ctx context.Context, req *pb.FormatDiskRequest) error {
	var pv *disk.Disk

	// Validate request first
	err := validateFormatDiskRequest(req)
	if err != nil {
		return err
	}
	if d.Status.Configured.Val && !req.Force {
		return status.Error(codes.FailedPrecondition, "Fail to format configured disk w/o force flag")
	}
	if err := d.MayModify(); err != nil {
		return nil
	}
	if err := d.Final.MakeGPT(ctx, ""); err != nil {
		return status.Error(codes.Internal, err.Error())
	}
	pvIdx := -1
	for idx, p := range req.Partitions {
		if p.Type == "" {
			p.Type = string(disk.PartTypeFS)
		}
		err := d.Final.MakePartition(ctx, uint(idx), uint64(p.StartBytes), uint64(p.SizeBytes),
			disk.PartType(p.Type), "")
		if err != nil {
			return status.Error(codes.Internal, err.Error())
		}
	}
	// If pv creation is not requested we are done
	if pvIdx == -1 {
		return nil
	}

	disks, err := disk.AllDisks(true, false)
	if err != nil {
		return status.Error(codes.Internal, "fail to update disks cache, err:"+err.Error())
	}

	for _, dsk := range disks {
		if dsk.Kind == disk.MediaKindPartition && dsk.Partition == uint(pvIdx) && dsk.PartDisk == d.Final.Name {
			pv = dsk
			break
		}
	}
	if pv == nil {
		return status.Error(codes.Internal, "fail to find PV")
	}
	// At this point we have valid partition for PV creation
	if err = lvm.CreatePV(ctx, pv.DevPath); err != nil {
		return status.Error(codes.Internal, "fail to create PV, err:"+err.Error())
	}

	tags := DefaultVGTags
	for k, v := range req.Labels {
		tags = append(tags, DiskmanLabel(k, v))
	}
	return lvm.CreateVG(ctx, pv.DevPath, VGPrefix+d.ID, tags)
}

func (d *Disk) Marshal() *pb.Disk {
	// FIXME implement this method
	return &pb.Disk{}
}

func (d *Disk) getVolumeByName(name string) *Volume {
	// FIXME implement this method
	return nil
}

func (d *Disk) createVolume(ctx context.Context, spec *pb.VolumeSpec) error {
	// FIXME implement this method
	return nil
}

func (d *Disk) deleteVolume(ctx context.Context, ID string) error {
	// FIXME implement this method
	return nil
}
