package diskmanager

import (
	"context"
	"strconv"
	"sync"
	"time"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	grpc_middleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpc_ctxtags "github.com/grpc-ecosystem/go-grpc-middleware/tags"
	grpc_opentracing "github.com/grpc-ecosystem/go-grpc-middleware/tracing/opentracing"
	"github.com/opentracing/opentracing-go"
	"go.uber.org/zap"
	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"
	"google.golang.org/grpc/peer"
)

type requestIDKey struct{}

// DefaultXRequestIDKey is metadata key name for request ID
const (
	DefaultXRequestIDKey      = "x-request-id"
	DefaultXClientIDKey       = "x-client-id"
	DefaultXClientIDCompatKey = "client_id"
)

var (
	lastID uint64 = 0
	mu     sync.Mutex
)

func newRequestID() string {
	mu.Lock()
	ret := lastID
	lastID = lastID + 1
	mu.Unlock()
	return strconv.FormatUint(ret, 16)
}

func RequestIDFromContext(ctx context.Context) string {
	md, ok := metadata.FromIncomingContext(ctx)
	if !ok {
		return newRequestID()
	}

	header, ok := md[DefaultXRequestIDKey]
	if !ok || len(header) == 0 {
		return newRequestID()
	}

	requestID := header[0]
	if requestID == "" {
		return newRequestID()
	}
	return requestID
}

func CliendIDFromContext(ctx context.Context) string {
	md, ok := metadata.FromIncomingContext(ctx)
	if ok {
		if val, ok := md[DefaultXClientIDKey]; ok {
			return val[0]
		}
		if val, ok := md[DefaultXClientIDCompatKey]; ok {
			return val[0]
		}
	}
	p, ok := peer.FromContext(ctx)
	if ok {
		return p.Addr.String()
	}
	return ""
}

func addContext(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {
	clog := ilog.Log().With(
		zap.String("client",
			CliendIDFromContext(ctx)),
		zap.String("req_id", RequestIDFromContext(ctx)),
		zap.String("method", info.FullMethod))

	clog.Info("start")
	ts := time.Now()
	resp, err := handler(ctx, req)
	clog.Info("finish", zap.Duration("duration", time.Since(ts)), zap.Error(err))
	return resp, err
}

func UnaryServerInterceptor() grpc.UnaryServerInterceptor {
	return func(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (resp interface{}, err error) {
		return addContext(ctx, req, info, handler)
	}
}

func NewGRPCServer() *grpc.Server {
	opts := []grpc_opentracing.Option{
		grpc_opentracing.WithTracer(opentracing.GlobalTracer()),
	}
	ServerOpts := []grpc.ServerOption{
		grpc_middleware.WithUnaryServerChain(
			grpc_ctxtags.UnaryServerInterceptor(grpc_ctxtags.WithFieldExtractor(grpc_ctxtags.CodeGenRequestFieldExtractor)),
			grpc_opentracing.UnaryServerInterceptor(opts...),
			UnaryServerInterceptor()),
	}
	return grpc.NewServer(ServerOpts...)
}
