package diskmanager

import (
	"context"
	"path"
	"sort"

	"go.uber.org/zap"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/disk"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/lvm"
)

type VgMap struct {
	uuid string
	vg   *lvm.VolumeGroup
	lvs  []*lvm.LogicalVolume
	pvs  []*lvm.PhysicalVolume
}

type DiskMap struct {
	wwn   string
	final *disk.Disk
	parts []*disk.Disk
	pv    *lvm.PhysicalVolume
	vg    *lvm.VolumeGroup
	lvs   []*lvm.LogicalVolume
}

func NewAllVgMaps(ctx context.Context) (map[string]*VgMap, error) {
	vmap := make(map[string]*VgMap)
	// TODO Fetch info in fault tolerant mode here
	lvs, err := lvm.ListLV(ctx, "")
	if err != nil {
		return nil, err
	}
	vgs, err := lvm.ListVG(ctx, "")
	if err != nil {
		return nil, err
	}
	pvs, err := lvm.ListPV(ctx, "")
	if err != nil {
		return nil, err
	}

	for _, vg := range vgs {
		vmap[vg.UUID] = &VgMap{vg: vg, uuid: vg.UUID}
	}
	for _, lv := range lvs {
		vmap[lv.UUID].lvs = append(vmap[lv.VgUUID].lvs, lv)
	}
	for _, pv := range pvs {
		vmap[pv.UUID].pvs = append(vmap[pv.VgUUID].pvs, pv)
	}
	return vmap, nil
}

func NewDiskMap(ctx context.Context) (map[string]*DiskMap, error) {
	ll := ilog.Log()
	// FIXME: AllDisk collector should be more reliable for broken disks
	dl, err := disk.AllDisks(true, false)
	if err != nil {
		return nil, err
	}
	disks := make(map[string]*disk.Disk)
	dmap := make(map[string]*DiskMap)
	for _, d := range dl {
		disks[d.Name] = d
		if d.IsFinal {
			dmap[d.Name] = &DiskMap{final: d}
		}
	}
	for _, d := range disks {
		if d.Kind != disk.MediaKindPartition {
			continue
		}
		if dm, ok := dmap[d.PartDisk]; ok {
			dm.parts = append(dm.parts, d)
		}
	}

	vmap, err := NewAllVgMaps(ctx)
	if err != nil {
		return nil, err
	}
	for _, g := range vmap {
		ok := false
		for _, tag := range g.vg.Tags {
			if tag == TagOwner {
				ok = true
				break
			}
		}
		if !ok {
			ll.Debug("Ignore unknown vg", zap.Any("vg", g.vg))
			continue
		}
		if len(g.pvs) != 1 {
			ll.Debug("Ignore vgmap, bad pvnum", zap.Any("VgMap", g))
			continue
		}
		pvName := path.Base(g.pvs[0].Name)
		pv, ok := disks[pvName]
		if !ok {
			ll.Error("Cant not find PV", zap.String("name", pvName))
			continue
		}
		if pv.Kind != disk.MediaKindPartition {
			ll.Debug("Unedpected pv kind, want partition", zap.Any("pv", pv))
			continue
		}
		dm, ok := dmap[pv.PartDisk]
		if !ok {
			ll.Debug("Can not find final device for", zap.Any("pv", pv))
			continue
		}
		dm.vg = g.vg
		dm.pv = g.pvs[0]
		dm.lvs = g.lvs
	}
	return dmap, nil
}

func NewDisk(dm *DiskMap) *Disk {
	parts := dm.parts
	sort.SliceStable(parts, func(i, j int) bool {
		return parts[i].PartStart < parts[j].PartStart
	})

	d := Disk{
		ID:    dm.final.WWN,
		Final: dm.final,
		Parts: parts,
		Vg:    dm.vg,
		Pv:    dm.pv,
		Lvs:   dm.lvs,
	}
	d.Status.Configured.Set(false, "Not yet configured")
	if dm.vg != nil {
		d.Status.Configured.Set(true, "")
		d.Status.Ready.Set(true, "")
	}
	return &d
}

func (d *Disk) syncState() {
	if d.Status.Absent.Val {
		d.Status.Configured.Set(false, "Disk is absent")
		d.Status.Ready.Set(false, "Disk is absent")
		return
	}
	if d.Status.Error.Val {
		d.Status.Ready.Set(false, "Error detected")
		return
	}
	if d.Status.Configured.Val {
		d.Status.Ready.Set(true, "")
	}
}
func (d *Disk) MergeState(nd *Disk) {
	d.Vg = nd.Vg
	d.Pv = nd.Pv
	d.Lvs = nd.Lvs
	d.Parts = nd.Parts
	// If we have new state to merge that we disk is found
	d.Status.Absent.Set(false, "")
	d.Status.Configured = nd.Status.Configured
	// If disk is in error state, we should not change that
	if nd.Status.Error.Val {
		d.Status.Error.Set(true, nd.Status.Error.Msg)
		return
	}
	d.syncState()
}

func NewDiskCache(ctx context.Context) (map[string]*Disk, error) {
	disks, err := NewDiskMap(ctx)
	if err != nil {
		return nil, err
	}
	cache := make(map[string]*Disk)
	for _, dm := range disks {
		cache[dm.final.WWN] = NewDisk(dm)
	}
	return cache, nil
}

func (dm *Diskmanager) updateCache(ctx context.Context) error {
	return status.Error(codes.Unimplemented, "implement me")
}
