package diskmanager

import (
	"context"
	"encoding/hex"
	"net"
	"os"
	"strings"
	"sync"
	"time"

	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/proto"
	opentracing "github.com/opentracing/opentracing-go"
	"go.uber.org/zap"
	"google.golang.org/grpc"

	pb "a.yandex-team.ru/infra/rsm/diskmanager/api"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/xerrors"
)

type Diskmanager struct {
	//static fields
	id             string
	pid            int
	startTime      time.Time
	grpcServer     *grpc.Server
	grpcListener   *net.Listener
	log            *zap.Logger
	yasmProdURL    string
	yasmProdCType  string
	yasmProdProj   string
	yasmPushPeriod uint32

	fstrimPeriod     uint32
	fstrimWaitFactor uint32
	quotaSyncPeriod  uint32

	// dynamic fields
	mux      sync.Mutex
	cacheTS  time.Time
	cacheGen uint64
	disks    map[string]*Disk
	volumes  map[string]*Volume

	stmux    sync.Mutex
	pubStats pb.DaemonGetStatResponse
	errChan  <-chan error
}

func NewDiskmanager(o Options) (*Diskmanager, error) {
	d := &Diskmanager{
		yasmProdURL:      o.YasmProdURL,
		yasmProdCType:    o.YasmProdCType,
		yasmProdProj:     o.YasmProdProj,
		yasmPushPeriod:   o.YasmPushPeriod,
		fstrimPeriod:     o.FstrimPeriod,
		fstrimWaitFactor: o.FstrimWaitFactor,
		quotaSyncPeriod:  o.QuotaSyncPeriod,
	}
	uid := uuid.Must(uuid.NewV4())
	d.id = hex.EncodeToString(uid.Bytes())[:16]
	d.pid = os.Getpid()
	d.startTime = time.Now()

	d.log = ilog.Log().With(zap.String("service", "diskmanager"))
	d.log.Info("init", zap.Any("config", o))

	if strings.ContainsRune(o.GrpcEndpoint, ':') {
		ap := strings.Split(o.GrpcEndpoint, ":")
		if len(ap) != 2 {
			return nil, xerrors.Errorf("Bad address pair: %v", ap)
		}
		if ap[0] != "unix" {
			return nil, xerrors.Errorf("Unsupported proto: %v", ap[0])
		}
		o.GrpcEndpoint = ap[1]
	}
	if o.GrpcEndpoint == "" {
		return nil, xerrors.Errorf("Empty grpc endpoint")
	}
	lis, err := net.Listen("unix", o.GrpcEndpoint)
	if err != nil {
		return nil, err
	}
	d.grpcListener = &lis
	d.grpcServer = NewGRPCServer()

	// Dummy span to announce service existence
	sp := opentracing.StartSpan("Diskmanager.Init")
	sp.SetTag("component", "diskmanager")
	sp.SetTag("span.kind", "server")
	defer sp.Finish()

	ilog.Log().Info("started",
		zap.Int("Pid", d.pid), zap.String("ID", d.id),
		zap.String("Verion", buildinfo.Info.ProgramVersion))

	pb.RegisterDiskManagerServer(d.grpcServer, d)

	return d, nil
}

func (dm *Diskmanager) Serve() error {
	return dm.grpcServer.Serve(*dm.grpcListener)
}

func (dm *Diskmanager) Stop(reason string) {
	dm.grpcServer.Stop()

	ilog.Log().Info("Diskmanager service stopped",
		zap.Int("Pid", dm.pid), zap.String("ID", dm.id),
		zap.String("Verion", buildinfo.Info.ProgramVersion),
		zap.String("reason", reason))

}

func (dm *Diskmanager) ListDisks(ctx context.Context, in *pb.ListDisksRequest) (*pb.ListDisksResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sListDisks(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) FormatDisk(ctx context.Context, in *pb.FormatDiskRequest) (*pb.FormatDiskResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sFormatDisk(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}
func (dm *Diskmanager) ListVolumes(ctx context.Context, in *pb.ListVolumesRequest) (*pb.ListVolumesResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sListVolumes(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) CreateVolume(ctx context.Context, in *pb.CreateVolumeRequest) (*pb.CreateVolumeResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sCreateVolume(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) DeleteVolume(ctx context.Context, in *pb.DeleteVolumeRequest) (*pb.DeleteVolumeResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sDeleteVolume(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) MountVolume(ctx context.Context, in *pb.MountVolumeRequest) (*pb.MountVolumeResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sMountVolume(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) UmountVolume(ctx context.Context, in *pb.UmountVolumeRequest) (*pb.UmountVolumeResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sUmountVolume(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) SetIOLimitVolume(ctx context.Context, in *pb.SetIOLimitRequest) (*pb.SetIOLimitResponse, error) {
	dm.mux.Lock()
	repl, err := dm.sSetIOLimitVolume(ctx, in)
	dm.mux.Unlock()
	return repl, GRPCErr(err)
}

func (dm *Diskmanager) DaemonGetStat(ctx context.Context, in *pb.DaemonGetStatRequest) (*pb.DaemonGetStatResponse, error) {
	var repl *pb.DaemonGetStatResponse
	dm.stmux.Lock()
	repl = proto.Clone(&dm.pubStats).(*pb.DaemonGetStatResponse)
	dm.stmux.Unlock()
	return repl, nil
}

func (dm *Diskmanager) DaemonUpdateCache(ctx context.Context, in *pb.DaemonUpdateCacheRequest) (*pb.DaemonUpdateCacheResponse, error) {
	var err error
	dm.mux.Lock()
	err = dm.updateCache(ctx)
	dm.mux.Unlock()
	return &pb.DaemonUpdateCacheResponse{}, GRPCErr(err)
}

func (dm *Diskmanager) ServerInfo(ctx context.Context, in *pb.ServerInfoRequest) (*pb.ServerInfoResponse, error) {
	return &pb.ServerInfoResponse{
		Version:   buildinfo.Info.ProgramVersion,
		Id:        dm.id,
		MainPid:   int64(dm.pid),
		StartTime: utils.TimeFormatPB(dm.startTime)}, nil
}

func (dm *Diskmanager) DaemonSetLoglevel(ctx context.Context, in *pb.DaemonSetLoglevelRequest) (*pb.DaemonSetLoglevelResponse, error) {
	switch in.Verbosity {
	case 0:
		ilog.SetLevel(zap.ErrorLevel)
	case 1:
		ilog.SetLevel(zap.WarnLevel)
	case 2:
		ilog.SetLevel(zap.InfoLevel)
	default:
		ilog.SetLevel(zap.DebugLevel)
	}
	return &pb.DaemonSetLoglevelResponse{}, nil
}

func (dm *Diskmanager) getDisk(ID string) *Disk {
	if d, ok := dm.disks[ID]; ok {
		return d
	}
	return nil
}

func (dm *Diskmanager) getVolume(ID string) *Volume {
	if v, ok := dm.volumes[ID]; ok {
		return v
	}
	return nil
}

func (dm *Diskmanager) notifyRuntimeError(err error) {

	dm.stmux.Lock()
	dm.pubStats.RuntimeErrors++
	dm.stmux.Unlock()
	//dm.errChan <- err
}
