package diskmanager

import (
	"context"

	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	pb "a.yandex-team.ru/infra/rsm/diskmanager/api"
)

func (dm *Diskmanager) sListDisks(ctx context.Context, in *pb.ListDisksRequest) (*pb.ListDisksResponse, error) {
	pbd := make([]*pb.Disk, 0, len(dm.disks))
	for _, d := range dm.disks {
		// TODO check disks here before report it to list
		pbd = append(pbd, d.Marshal())
	}
	return &pb.ListDisksResponse{Disks: pbd}, nil
}

func (dm *Diskmanager) sFormatDisk(ctx context.Context, in *pb.FormatDiskRequest) (*pb.FormatDiskResponse, error) {
	d := dm.getDisk(in.DiskId)
	if d == nil {
		return nil, status.Error(codes.NotFound, "Disk not found")
	}
	err := d.format(ctx, in)
	if err != nil {
		return nil, GRPCErr(err)
	}
	err = dm.updateCache(ctx)
	if err != nil {
		return nil, GRPCErr(err)
	}
	d = dm.getDisk(in.DiskId)
	if d == nil {
		return nil, status.Error(codes.NotFound, "Disk not found")
	}
	return &pb.FormatDiskResponse{Disk: d.Marshal()}, nil
}

func (dm *Diskmanager) sListVolumes(ctx context.Context, in *pb.ListVolumesRequest) (*pb.ListVolumesResponse, error) {
	pbv := make([]*pb.Volume, 0, len(dm.volumes))
	for _, v := range dm.volumes {
		// TODO check disks here before report it to list
		pbv = append(pbv, v.Marshal())
	}
	return &pb.ListVolumesResponse{Volumes: pbv}, nil
}

func (dm *Diskmanager) sCreateVolume(ctx context.Context, in *pb.CreateVolumeRequest) (*pb.CreateVolumeResponse, error) {
	d := dm.getDisk(in.VolumeSpec.DiskId)
	if d == nil {
		return nil, status.Error(codes.NotFound, "Disk not found")
	}
	// Check that we already have volume
	v := d.getVolumeByName(in.VolumeSpec.Name)
	if v != nil {
		err := v.cmpSpec(in.VolumeSpec)
		if err != nil {
			return nil, err
		}
		return &pb.CreateVolumeResponse{Volume: v.Marshal()}, nil
	}
	err := d.createVolume(ctx, in.VolumeSpec)
	if err != nil {
		return nil, GRPCErr(err)
	}
	err = dm.updateCache(ctx)
	if err != nil {
		return nil, GRPCErr(err)
	}
	v = d.getVolumeByName(in.VolumeSpec.Name)
	if v == nil {
		return nil, status.Error(codes.NotFound, "volume dissapear under us")
	}
	return &pb.CreateVolumeResponse{Volume: v.Marshal()}, nil
}

func (dm *Diskmanager) sDeleteVolume(ctx context.Context, in *pb.DeleteVolumeRequest) (*pb.DeleteVolumeResponse, error) {
	v := dm.getVolume(in.VolumeId)

	if v == nil {
		return &pb.DeleteVolumeResponse{}, nil
	}
	d := v.Parent
	err := d.deleteVolume(ctx, v.ID)
	if err != nil {
		// Convert error to grpc one here
		gerr := err
		return nil, gerr
	}
	return &pb.DeleteVolumeResponse{}, nil
}

func (dm *Diskmanager) sMountVolume(ctx context.Context, in *pb.MountVolumeRequest) (*pb.MountVolumeResponse, error) {
	v := dm.getVolume(in.VolumeId)
	if v == nil {
		return nil, status.Error(codes.NotFound, "not such volume")
	}
	if err := v.doMount(ctx, in.MountPath); err != nil {
		return nil, GRPCErr(err)
	}
	return &pb.MountVolumeResponse{}, nil
}

func (dm *Diskmanager) sUmountVolume(ctx context.Context, in *pb.UmountVolumeRequest) (*pb.UmountVolumeResponse, error) {
	v := dm.getVolume(in.VolumeId)
	if v == nil {
		return nil, status.Error(codes.NotFound, "not such volume")
	}
	if v.Device.MInfo.Mountpoint == "" {
		return &pb.UmountVolumeResponse{}, nil
	}
	if err := v.doUnmount(ctx); err != nil {
		// Convert error to grpc one here
		gerr := err
		return nil, gerr
	}
	return &pb.UmountVolumeResponse{}, nil
}

func (dm *Diskmanager) sSetIOLimitVolume(ctx context.Context, in *pb.SetIOLimitRequest) (*pb.SetIOLimitResponse, error) {
	v := dm.getVolume(in.Id)
	if v == nil {
		return nil, status.Error(codes.NotFound, "not such volume")
	}
	if err := v.setIOLimit(ctx, in.Iolimit); err != nil {
		// Convert error to grpc one here
		gerr := err
		return nil, gerr
	}
	return &pb.SetIOLimitResponse{}, nil
}

func (dm *Diskmanager) sDaemonUpdateCache(ctx context.Context, in *pb.DaemonUpdateCacheRequest) (*pb.DaemonUpdateCacheResponse, error) {
	err := dm.updateCache(ctx)
	if err != nil {
		// Convert error to grpc one here
		gerr := err
		return nil, gerr
	}
	return &pb.DaemonUpdateCacheResponse{}, nil
}
