package diskmanager

import (
	"io"

	"github.com/opentracing/opentracing-go"
	jaeger "github.com/uber/jaeger-client-go"
	jaegercfg "github.com/uber/jaeger-client-go/config"
	jaegerlog "github.com/uber/jaeger-client-go/log"
	"go.uber.org/zap"
)

type loggingReporter struct {
	logger zap.Logger
}

// NewLoggingReporter creates a reporter that logs all reported spans to provided logger.
func NewLoggingReporter(logger zap.Logger) jaeger.Reporter {
	return &loggingReporter{logger}
}

// Report implements Report() method of Reporter by logging the span to the logger.
func (r *loggingReporter) Report(span *jaeger.Span) {
	r.logger.Info("Reporting span",
		zap.String("ctx", span.String()),
		zap.String("opName", span.OperationName()))
	//		zap.Array("tags", span.tags))
}

// Close implements Close() method of Reporter by doing nothing.
func (r *loggingReporter) Close() {
	// no-op
}

func InitTracer(logger *zap.Logger, cfg jaegercfg.Configuration) (io.Closer, error) {

	if cfg.ServiceName == "" {
		cfg = jaegercfg.Configuration{
			ServiceName: "diskmanager",
			Sampler: &jaegercfg.SamplerConfig{
				Type:  jaeger.SamplerTypeConst,
				Param: 1,
			},
			Reporter: &jaegercfg.ReporterConfig{
				CollectorEndpoint: "http://dmws02.man.yp-c.yandex.net:14268/api/traces?format=jaeger.thrift",
				LogSpans:          true,
			},
		}
		logger.Debug("Generate tracer", zap.Any("config", cfg))
	}
	jLogger := jaegerlog.StdLogger
	tracer, closer, err := cfg.NewTracer(
		jaegercfg.Logger(jLogger),
	)
	if err != nil {
		return nil, err
	}
	logger.Info("Init tracer", zap.String("ServiceName", cfg.ServiceName))
	opentracing.SetGlobalTracer(tracer)
	return closer, nil
}
