package disk

import (
	"bytes"
	"context"
	"fmt"
	"os/exec"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
	opentracing "github.com/opentracing/opentracing-go"
	"github.com/opentracing/opentracing-go/ext"
	"go.uber.org/zap"
)

var (
	diskTag = opentracing.Tag{Key: "component", Value: "interal/disk"}
)

func (d *Disk) MakeFs(ctx context.Context, fstype string, ownerUID uint, ownerGID uint) error {
	var cmd *exec.Cmd
	switch fstype {
	case "ext4":
		cmd = exec.Command("mkfs.ext4", "-q", "-F", "-E",
			fmt.Sprintf("root_owner=%d:%d", ownerUID, ownerGID), d.DevPath)
	case "xfs":
		cmd = exec.Command("mkfs.xfs", "-q", "-f", d.DevPath)
	default:
		return fmt.Errorf("unknown filesystem: %s", fstype)
	}
	return runCmd(ctx, cmd)
}

func (d *Disk) CheckFs(ctx context.Context, fstype string, force bool) error {
	var cmd *exec.Cmd
	switch fstype {
	case "ext4":
		cmd = exec.Command("fsck.ext4", "-f")
		if force {
			cmd.Args = append(cmd.Args, "-y")
		} else {
			cmd.Args = append(cmd.Args, "-p")
		}
	case "xfs":
		cmd = exec.Command("fsck.xfs")
	default:
		return fmt.Errorf("fsck: unknown filesystem: %s", fstype)
	}
	cmd.Args = append(cmd.Args, d.DevPath)
	return runCmd(ctx, cmd)
}

func doPartprobe(ctx context.Context, name string) error {
	c := exec.Command("partprobe", name)
	return runCmd(ctx, c)
}

func (d *Disk) Wipe(ctx context.Context) error {
	c := exec.Command("wipefs", "-a", d.DevPath)
	err := runCmd(ctx, c)
	if err != nil {
		return err
	}
	if d.IsFinal {
		err = doPartprobe(ctx, d.DevPath)
	}
	return err
}

func (d *Disk) MakeGPT(ctx context.Context, guid string) error {
	// Use random id, if not provided
	if guid == "" {
		guid = "R"
	}
	c := exec.Command("sgdisk", "-o", "-g", "-U", guid, d.DevPath)
	err := runCmd(ctx, c)
	if err != nil {
		return err
	}
	return doPartprobe(ctx, d.DevPath)

}

func (d *Disk) MakePartition(ctx context.Context, idx uint, start uint64, size uint64, ptype PartType, name string) error {
	if name == "" {
		name = fmt.Sprintf("part-%d", idx)
	}
	c := exec.Command("sgdisk",
		"-n", fmt.Sprintf("%d:%d:%d", idx, start>>9, ((start+size)>>9)-1),
		"-c", fmt.Sprintf("%d:'%s'", idx, name),
		"-t", fmt.Sprintf("%d:%s", idx, string(ptype)),
		d.DevPath)
	err := runCmd(ctx, c)
	if err != nil {
		return err
	}
	return doPartprobe(ctx, d.DevPath)
}

func (d *Disk) InstallGrub(ctx context.Context) error {
	c := exec.Command("grub-install", d.DevPath)
	return runCmd(ctx, c)
}

func runCmd(ctx context.Context, c *exec.Cmd) error {
	ll := ilog.Log()

	sp, _ := opentracing.StartSpanFromContext(ctx, c.Args[0], diskTag, ext.SpanKindRPCClient)
	defer sp.Finish()

	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr

	err := c.Run()

	lf := []zap.Field{
		zap.String("cmd", c.String()), zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()),
	}
	if err != nil {
		utils.LogSpanError(sp, err)
		ll.Error("cmd fail", lf...)
		return err
	}
	ll.Debug("exec", lf...)
	return nil
}
