package disk_test

import (
	"context"
	"fmt"
	"os"
	"os/user"
	"path"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/disk"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/loop"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/sysfs"
)

type noopSpan struct{}

const (
	devSize = uint64(128 << 20)
	devName = "test.img"
)

func requireRoot(t *testing.T) {
	u, _ := user.Current()
	assert.NotNil(t, u)
	if u.Uid != "0" {
		t.Skipf("Test requires root privileges, current uid is: %s ", u.Uid)
	}
}

func TestFsLifeCycle(t *testing.T) {
	requireRoot(t)

	ctx := context.Background()
	loop, err := loop.CreateTMPLoopDevice(devSize, devName)
	require.NoError(t, err)
	defer loop.Close()

	var tests = []struct {
		fsType string
		uid    uint
		gid    uint
		ok     bool
		opt1   []string
		opt2   []string
	}{
		{"ext4", 0, 0, true, []string{"barrier=1"}, []string{"discard"}},
		{"ext4", 0, 0, true, []string{}, []string{"discard", "quota", "rw"}},
		{"xfs", 0, 0, true, nil, nil},
		{"ntfs", 0, 0, false, []string{}, []string{}},
	}
	for _, tt := range tests {
		t.Run(tt.fsType, func(t *testing.T) {
			d, err := disk.NewDisk(path.Base(loop.Path()))
			require.NoError(t, err)

			err = d.MakeFs(ctx, tt.fsType, tt.uid, tt.gid)
			if tt.ok {
				require.NoErrorf(t, err, "Unexpected error for %s", tt.fsType)
			} else {
				require.Errorf(t, err, "Unexpected success for %s", tt.fsType)
			}
			// Check that we can create fs on device which already has one
			err = d.MakeFs(ctx, tt.fsType, tt.uid, tt.gid)
			if tt.ok {
				require.NoErrorf(t, err, "Unexpected error for %s", tt.fsType)
			} else {
				require.Errorf(t, err, "Unexpected success for %s", tt.fsType)
				return
			}
			err = os.Mkdir(tt.fsType, 0755)
			require.NoError(t, err)
			defer os.Remove(tt.fsType)

			err = d.MountFs(ctx, tt.fsType, tt.fsType, 0, tt.opt1)
			require.NoError(t, err)

			err = d.RemountFs(ctx, tt.fsType, 0, tt.opt2)
			assert.NoError(t, err)

			err = d.UnmountFs(ctx, tt.fsType, 0)
			assert.NoError(t, err)

		})
	}
}

func TestExt4Error(t *testing.T) {
	requireRoot(t)

	ctx := context.Background()
	loop, err := loop.CreateTMPLoopDevice(devSize, devName)
	require.NoError(t, err)
	defer loop.Close()

	d, err := disk.NewDisk(path.Base(loop.Path()))
	require.NoError(t, err)

	err = d.MakeFs(ctx, "ext4", 0, 0)
	require.NoErrorf(t, err, "Unexpected error for %s", "ext4")

	err = os.Mkdir("ext4", 0755)
	require.NoError(t, err)
	defer os.Remove("ext4")

	err = d.MountFs(ctx, "ext4", "ext4", 0, nil)
	require.NoError(t, err)

	// Trigger fserror and update disk info
	fobj := sysfs.FS.Object("ext4").SubObject(d.Name)
	trigger := fobj.Attribute("trigger_fs_error")
	ilog.Log().Info("Trigger fs error", zap.String("sysfspath", string(trigger)))
	_ = trigger.WriteInt(1)

	d2, err := disk.NewDisk(path.Base(loop.Path()))
	if err != nil {
		_ = d.UnmountFs(ctx, "ext4", 0)
		t.Fatalf("fail to update device after fserror")
	}
	assert.NotZerof(t, d2.FsErrCount, "FsErrCount should not be zero")
	assert.Equal(t, d2.Error, disk.ErrIO, "disk.Error not much after trigger_fs_error")
	err = d.UnmountFs(ctx, "ext4", 0)
	assert.NoError(t, err)
}

func doPart(t *testing.T, ctx context.Context, d *disk.Disk, idx uint, start uint64, size uint64) *disk.Disk {
	err := d.MakePartition(ctx, idx, start, size, disk.PartTypeFS, "")
	require.NoError(t, err)

	pname := fmt.Sprintf("%sp%d", d.Name, idx)
	p, err := disk.NewDisk(pname)
	require.NoErrorf(t, err, "Can not find partition %s", pname)
	require.Equal(t, size, p.Size, "Unexpected size for dev:%s", d.Name)
	require.Equal(t, start, p.PartStart, "Unexpected size for dev:%s", d.Name)
	require.Equal(t, disk.MediaKindPartition, p.Kind)
	require.Equal(t, []string{d.Name}, p.Backends)
	require.Equal(t, idx, p.Partition)
	require.Equal(t, d.Name, p.PartDisk)

	return p
}

func doFormat(t *testing.T, ctx context.Context, dev string) *disk.Disk {
	d, err := disk.NewDisk(dev)
	require.NoError(t, err)
	err = d.MakeGPT(ctx, "")
	require.NoError(t, err)
	return doPart(t, ctx, d, 1, uint64(16<<20), uint64(16<<20))
}

func TestPartCreation(t *testing.T) {
	requireRoot(t)

	var tests = []struct {
		fstype string
	}{
		{"ext4"},
		{"xfs"},
	}
	for _, tt := range tests {
		t.Run(tt.fstype, func(t *testing.T) {
			ctx := context.Background()
			loop, err := loop.CreateTMPLoopDevice(devSize, tt.fstype+"-"+devName)
			require.NoError(t, err)
			defer loop.Close()

			d, err := disk.NewDisk(path.Base(loop.Path()))
			require.NoError(t, err)
			p := doFormat(t, ctx, d.Name)
			err = p.MakeFs(ctx, tt.fstype, 0, 0)
			require.NoError(t, err)

			// Wipe disks data, and repeate action
			err = d.Wipe(ctx)
			require.NoError(t, err)
			p = doFormat(t, ctx, d.Name)

			err = p.MakeFs(ctx, tt.fstype, 0, 0)
			require.NoError(t, err)
			err = p.CheckFs(ctx, tt.fstype, false)
			require.NoError(t, err)

			//create second partition
			p2 := doPart(t, ctx, d, 2, uint64(32<<20), uint64(32<<20))
			err = p2.MakeFs(ctx, tt.fstype, 0, 0)
			require.NoError(t, err)
			// Wipe partition and check that other partitions are OK
			err = p2.Wipe(ctx)
			require.NoError(t, err)

			err = p.CheckFs(ctx, tt.fstype, false)
			require.NoError(t, err)
		})
	}
}
