package disk_test

import (
	"testing"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/disk"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/sysfs"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"
)

func TestOneDisk(t *testing.T) {
	l := ilog.Log()

	var tests = []struct {
		dev      string
		kind     disk.MediaKind
		isFinal  bool
		backends []string
	}{
		{"loop0", disk.MediaKindLoopdev, true, nil},
		{"sda", disk.MediaKindScsiDisk, true, nil},
		{"ram0", disk.MediaKindLoopdev, true, nil},
		{"vda", disk.MediaKindVirtIO, true, nil},
		{"vdb", disk.MediaKindVirtIO, true, nil},
		{"sda1", disk.MediaKindPartition, false, []string{"sda"}},
		{"vdb1", disk.MediaKindPartition, false, []string{"vdb"}},
	}
	for _, tt := range tests {
		t.Run(tt.dev, func(t *testing.T) {
			if !sysfs.Class.Object("block").SubObject(tt.dev).Exists() {
				t.Skipf("disk %s is absent", tt.dev)
				return
			}
			d, err := disk.NewDisk(tt.dev)
			require.NoError(t, err)
			require.NotNil(t, d)
			l.Info("Validate", zap.String("name", tt.dev))
			assert.Equal(t, tt.dev, d.Name)
			assert.FileExists(t, "/dev/"+d.Name)
			assert.Equal(t, d.Kind, tt.kind)
			assert.Equal(t, d.IsFinal, tt.isFinal)
			if tt.isFinal {
				assert.NotEmptyf(t, d.WWN, "Final dev %s must have unique WWN ID", d.Name)
			}
			if tt.kind == disk.MediaKindPartition {
				assert.Equal(t, d.Backends, tt.backends)
			}
		})
	}
}

func TestListDisks(t *testing.T) {
	l := ilog.Log()

	var tests = []struct {
		name   string
		noZero bool
		final  bool
	}{
		{"ListAll", false, false},
		{"ListAllValid", true, false},
		{"ListAllValidFinal", true, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			disks, err := disk.AllDisks(tt.noZero, tt.final)
			require.NoError(t, err)
			require.NotEmpty(t, disks)
			l.Info("found", zap.String("test_name", tt.name), zap.Int("num_disks", len(disks)))

			wwns := make(map[string]int)
			names := make(map[string]int)
			for idx, d := range disks {
				l.Info("Iterate", zap.Int("idx", idx), zap.Any("disk", d))
				require.NotEmpty(t, d.Name)
				require.FileExists(t, "/dev/"+d.Name)
				if tt.noZero {
					require.Truef(t, d.Size != 0, "device %s has zero size :%v", d.Name, d)
				}
				if tt.final {
					require.True(t, d.IsFinal, "device %s is not final %v", d.Name, d)
				}
				if found, ok := names[d.Name]; ok {
					t.Errorf("Duplicate name found for: %s, idx1: %d, idx2: %d map:%v", d.Name, idx, found, names)
				}
				names[d.Name] = idx

				if d.WWN != "" {
					if found, ok := wwns[d.WWN]; ok {
						t.Errorf("WWN collision, %s(%d) and %s(%d) has same WWN:%s",
							disks[found].Name, found, d.Name, idx, d.WWN)
					}
					wwns[d.WWN] = idx
				}
				switch d.Kind {
				case disk.MediaKindPartition:
					assert.Empty(t, d.WWN)
					assert.False(t, d.IsFinal)
					assert.NotEmpty(t, d.Backends)
				case disk.MediaKindMD, disk.MediaKindDeviceMapper:
					assert.NotEmpty(t, d.WWN)
					assert.False(t, d.IsFinal)
					assert.NotEmpty(t, d.Backends)
				case disk.MediaKindScsiDisk, disk.MediaKindVirtIO, disk.MediaKindNVME, disk.MediaKindRAMDev, disk.MediaKindLoopdev:
					assert.NotEmptyf(t, d.WWN, "Final dev %s must have unique WWN ID", d.Name)
					assert.True(t, d.IsFinal)
				}
			}
		})
	}
}
