package disk

import (
	"context"
	"syscall"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
	opentracing "github.com/opentracing/opentracing-go"
	"github.com/opentracing/opentracing-go/ext"
	"go.uber.org/zap"
)

func buildMountOpt(opts []string) (uintptr, string) {
	var flags uintptr = 0
	opt := ""

	for _, o := range opts {
		switch o {
		case "noatime":
			flags |= uintptr(syscall.MS_NOATIME)
		case "atime":
			flags &= ^uintptr(syscall.MS_NOATIME)
		case "nodev":
			flags |= uintptr(syscall.MS_NODEV)
		case "dev":
			flags &= ^uintptr(syscall.MS_NODEV)
		case "noexec":
			flags |= uintptr(syscall.MS_NOEXEC)
		case "exec":
			flags &= ^uintptr(syscall.MS_NOEXEC)
		case "nosuid":
			flags |= uintptr(syscall.MS_NOSUID)
		case "suid":
			flags &= uintptr(syscall.MS_NOSUID)
		case "nouser":
			flags |= ^uintptr(^syscall.MS_NOUSER)
		case "user":
			flags |= uintptr(^syscall.MS_NOUSER)
		case "ro":
			flags |= uintptr(syscall.MS_RDONLY)
		case "rw":
			flags &= ^uintptr(syscall.MS_RDONLY)
		case "relatime":
			flags |= uintptr(syscall.MS_RELATIME)
		case "norelatime":
			flags &= ^uintptr(syscall.MS_RELATIME)
		default:
			opt = opt + "," + o
		}
	}
	return flags, opt
}

func doMount(src string, mnt string, fsType string, flags uintptr, opts []string) error {
	nf, data := buildMountOpt(opts)
	flags |= nf

	ilog.Log().Debug("mount", zap.String("src", src),
		zap.String("mnt", mnt), zap.String("fstype", fsType),
		zap.Uint64("flags", uint64(flags)), zap.String("data", data),
		zap.Strings("origOpts", opts))

	return syscall.Mount(src, mnt, fsType, syscall.MS_MGC_VAL|flags, data)

}

func doUnmount(target string, flags int) error {
	ilog.Log().Debug("umount", zap.String("target", target),
		zap.Int("flags", flags))

	return syscall.Unmount(target, 0)
}

func (d *Disk) MountFs(ctx context.Context, fstype string, path string, flags uintptr, opts []string) error {
	sp, _ := opentracing.StartSpanFromContext(ctx, "mount", diskTag, ext.SpanKindRPCClient)
	defer sp.Finish()

	err := doMount(d.DevPath, path, fstype, flags, opts)
	if err != nil {
		utils.LogSpanError(sp, err)
	}
	return err
}

func (d *Disk) RemountFs(ctx context.Context, path string, flags uintptr, opts []string) error {
	sp, _ := opentracing.StartSpanFromContext(ctx, "remount", diskTag, ext.SpanKindRPCClient)
	defer sp.Finish()

	err := doMount(d.DevPath, path, "", flags|syscall.MS_REMOUNT, opts)
	if err != nil {
		utils.LogSpanError(sp, err)
	}
	return err
}

func (d *Disk) UnmountFs(ctx context.Context, target string, flags int) error {
	sp, _ := opentracing.StartSpanFromContext(ctx, "umount", diskTag, ext.SpanKindRPCClient)
	defer sp.Finish()

	return doUnmount(target, flags)
}
