package loop

import (
	"fmt"
	"io/ioutil"
	"os"

	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/cleanup"
	"a.yandex-team.ru/library/go/core/xerrors"
)

type TMPLoopDevice struct {
	lodev           Device
	backingFilePath string
}

// CreateLoopDevice returns a file-backed loop device.  The caller is
// responsible for calling `Close()` on the `*TMPLoopDevice` when done
// with it.
//
// CreateLoopDevice may panic if an error occurs during error recovery.
func CreateTMPLoopDevice(size uint64, name string) (device *TMPLoopDevice, err error) {
	var cleanup cleanup.Steps
	defer func() {
		if err != nil {
			cleanup.Unwind()
		}
	}()

	file, err := ioutil.TempFile("", name)
	if err != nil {
		fmt.Printf("Fail")
		return nil, xerrors.Errorf("losetup.TemFile.Create: %w", err)
	}
	// If anything goes wrong, remove the tempfile.
	cleanup.Add(func() error { return os.Remove(file.Name()) })
	if err = file.Close(); err != nil {
		return nil, xerrors.Errorf("losetup.close: %w", err)
	}

	if err = os.Truncate(file.Name(), int64(size)); err != nil {
		return nil, xerrors.Errorf("losetup.Truncate: %w", err)
	}
	lo, err := Attach(file.Name(), 0, false)
	if err != nil {
		return nil, xerrors.Errorf("losetup.Attach: %w", err)
	}
	cleanup.Add(func() error { return lo.Detach() })
	return &TMPLoopDevice{lo, file.Name()}, nil
}

func (d *TMPLoopDevice) Path() string {
	return d.lodev.Path()
}

func (d *TMPLoopDevice) String() string {
	return d.lodev.Path()
}

func (d *TMPLoopDevice) BackingPath() string {
	return d.backingFilePath
}

// Close detaches the loop device and removes the backing file.
func (d *TMPLoopDevice) Close() error {
	if err := d.lodev.Detach(); err != nil {
		return err
	}
	return os.Remove(d.backingFilePath)
}
