package lvm

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"os/exec"
	"regexp"
	"strings"

	"github.com/opentracing/opentracing-go"
	"github.com/opentracing/opentracing-go/ext"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
)

var (
	lvmNameRegexp = regexp.MustCompile("^[A-Za-z0-9_+.][A-Za-z0-9_+.-]*$")
	lvmTag        = opentracing.Tag{Key: "component", Value: "lvm"}
)

func ValidateGroupName(name string) error {
	if !lvmNameRegexp.MatchString(name) {
		return ErrInvalidVGName
	}
	return nil
}

func ValidateVolumeName(name string) error {
	if !lvmNameRegexp.MatchString(name) {
		return ErrInvalidLVName
	}
	return nil
}

func ValidateTag(tag string) error {
	if len(tag) > 1024 {
		return ErrTagInvalidLength
	}
	if !lvmNameRegexp.MatchString(tag) {
		return ErrTagHasInvalidChars
	}
	return nil
}

// Parse lvm output and try to find known patterns
// TODO: Find a beter way to get error code from lvm call
func isInsufficientSpace(err error) bool {
	return strings.Contains(strings.ToLower(err.Error()), "insufficient free space")
}

func isInsufficientDevices(err error) bool {
	return strings.Contains(err.Error(), "Insufficient suitable allocatable extents for logical volume")
}

func volumePath(vg string, name string) string {
	return fmt.Sprintf("/dev/%s/%s", vg, name)
}

func ListLV(ctx context.Context, listspec string) ([]*LogicalVolume, error) {
	args := []string{"--units=b", "--separator=<:XSEP:>", "--nosuffix", "--noheadings", "--nameprefixes", "-a",
		"-o", "lv_uuid,lv_name,lv_size,lv_attr,lv_kernel_major,lv_kernel_minor,lv_tags,vg_uuid,vg_name",
		listspec}

	out, err := runLvm(ctx, "lvs", args...)
	if err != nil {
		return nil, err
	}
	outStr := strings.TrimSpace(out)
	outLines := strings.Split(outStr, "\n")
	lvs := make([]*LogicalVolume, len(outLines))
	for i, line := range outLines {
		line = strings.TrimSpace(line)
		lv, err := parseLV(line, true)
		if err != nil {
			return nil, err
		}
		lvs[i] = lv
	}
	return lvs, nil
}

func ListVG(ctx context.Context, listspec string) ([]*VolumeGroup, error) {
	args := []string{"--units=b", "--separator=<:XSEP:>", "--nosuffix", "--noheadings", "--nameprefixes", "-a",
		"-o", "vg_uuid,vg_name,vg_size,vg_free,vg_sysid,vg_extent_size,vg_tags,lv_count",
		listspec}

	out, err := runLvm(ctx, "vgs", args...)
	if err != nil {
		return nil, err
	}
	outStr := strings.TrimSpace(out)
	outLines := strings.Split(outStr, "\n")
	vgs := make([]*VolumeGroup, len(outLines))
	for i, line := range outLines {
		line = strings.TrimSpace(line)
		vg, err := parseVG(line)
		if err != nil {
			return nil, err
		}
		vgs[i] = vg
	}
	return vgs, nil
}

func ListPV(ctx context.Context, listspec string) ([]*PhysicalVolume, error) {
	args := []string{"--units=b", "--separator=<:XSEP:>", "--nosuffix", "--noheadings", "--nameprefixes",
		"-o", "pv_name,pv_uuid,pv_size,pv_free,pv_tags,pe_start,dev_size,vg_uuid,vg_name",
		listspec}

	out, err := runLvm(ctx, "pvs", args...)
	if err != nil {
		return nil, err
	}
	outStr := strings.TrimSpace(out)
	outLines := strings.Split(outStr, "\n")
	pvs := make([]*PhysicalVolume, len(outLines))
	for i, line := range outLines {
		line = strings.TrimSpace(line)
		pv, err := parsePV(line)
		if err != nil {
			return nil, err
		}
		pvs[i] = pv
	}
	return pvs, nil
}

func LookupLV(ctx context.Context, vg string, name string) (*LogicalVolume, error) {
	lvs, err := ListLV(ctx, volumePath(vg, name))
	if err != nil {
		return nil, err
	}
	if len(lvs) == 0 {
		return nil, ErrLogicalVolumeNotFound
	}
	if len(lvs) != 1 {
		ilog.Log().Error("too many items, want 1", zap.Int("got", len(lvs)))
		return nil, ErrLogicalVolumeNotFound
	}

	lv := lvs[0]
	if lv.Name != name {
		ilog.Log().Error("Unexpected volume name", zap.String("want", name), zap.String("got", lv.Name))
		return nil, ErrLogicalVolumeNotFound
	}
	return lv, nil
}

func LookupVG(ctx context.Context, name string) (*VolumeGroup, error) {
	vgs, err := ListVG(ctx, name)
	if err != nil {
		return nil, err
	}
	if len(vgs) == 0 {
		return nil, ErrVolumeGroupNotFound
	}
	if len(vgs) != 1 {
		ilog.Log().Error("too many items, want 1", zap.Int("got", len(vgs)))
		return nil, ErrVolumeGroupNotFound
	}
	vg := vgs[0]
	if vg.Name != name {
		ilog.Log().Error("unexpected group name", zap.String("want", name), zap.String("got", vg.Name))
		return nil, ErrVolumeGroupNotFound
	}
	return vg, nil
}

func LookupPV(ctx context.Context, name string) (*PhysicalVolume, error) {
	pvs, err := ListPV(ctx, name)
	if err != nil {
		return nil, err
	}
	if len(pvs) == 0 {
		return nil, ErrPhysicalVolumeNotFound
	}

	if len(pvs) != 1 {
		ilog.Log().Error("too many items, want 1", zap.Int("got", len(pvs)))
		return nil, ErrPhysicalVolumeNotFound
	}
	pv := pvs[0]
	if pv.Name != name {
		ilog.Log().Error("unexpected pv name", zap.String("want", name), zap.String("got", pv.Name))
		return nil, ErrPhysicalVolumeNotFound
	}
	return pv, nil
}

func PVScan(ctx context.Context, dev string) error {
	args := []string{"--cache"}
	if dev != "" {
		args = append(args, dev)
	}
	_, err := runLvm(ctx, "pvscan", args...)
	return err
}

func VGScan(ctx context.Context, name string) error {
	args := []string{"--cache"}
	if name != "" {
		args = append(args, name)
	}
	_, err := runLvm(ctx, "vgscan", args...)
	return err
}

func CreateLV(ctx context.Context, vgName string, name string, size uint64, tags []string) error {
	if size == 0 {
		return errors.New("size must be greater than 0")
	}
	if err := ValidateVolumeName(name); err != nil {
		return err
	}
	args := []string{"-v", "--name", name, "-L", fmt.Sprintf("%db", size), vgName}
	for _, tag := range tags {
		if err := ValidateTag(tag); err != nil {
			return err
		}
		args = append(args, "--addtag", tag)
	}
	if _, err := runLvm(ctx, "lvcreate", args...); err != nil {
		if isInsufficientSpace(err) {
			return ErrNoSpace
		}
		if isInsufficientDevices(err) {
			return ErrTooFewDisks
		}
		return err
	}
	return nil
}

func CreateVG(ctx context.Context, pvName string, name string, tags []string) error {
	if err := ValidateGroupName(name); err != nil {
		return err
	}
	args := []string{"-v", name, pvName}
	for _, tag := range tags {
		if err := ValidateTag(tag); err != nil {
			return err
		}
		args = append(args, "--addtag", tag)
	}
	if _, err := runLvm(ctx, "vgcreate", args...); err != nil {
		return err
	}
	// Perform a best-effort scan to trigger a lvmetad cache refresh.
	// We ignore errors as for better or worse, the volume group now exists.
	// Without this lvmetad can fail to pickup newly created volume groups.
	// See https://bugzilla.redhat.com/show_bug.cgi?id=837599
	if err := PVScan(ctx, ""); err != nil {
		ilog.Log().Error("pvscan failed", zap.Error(err))
	}
	if err := VGScan(ctx, ""); err != nil {
		ilog.Log().Error("vgscan failed", zap.Error(err))
	}
	return nil
}

func CreatePV(ctx context.Context, name string) error {
	args := []string{"-v", name}
	if _, err := runLvm(ctx, "pvcreate", args...); err != nil {
		return err
	}
	// Perform a best-effort scan to trigger a lvmetad cache refresh.
	// We ignore errors as for better or worse, the volume group now exists.
	// Without this lvmetad can fail to pickup newly created volume groups.
	// See https://bugzilla.redhat.com/show_bug.cgi?id=837599
	if err := PVScan(ctx, ""); err != nil {
		ilog.Log().Error("pvscan failed", zap.Error(err))
	}
	return nil
}

func RemovePV(ctx context.Context, name string) error {
	_, err := runLvm(ctx, "pvremove", "-v", "-f", name)
	return err
}

func RemoveVG(ctx context.Context, name string) error {
	_, err := runLvm(ctx, "vgremove", "-v", "-f", name)
	return err
}

func RemoveLV(ctx context.Context, vg string, name string) error {
	_, err := runLvm(ctx, "lvremove", "-v", "-f", volumePath(vg, name))
	return err
}

func doChangeTags(ctx context.Context, cmd string, name string, addTags []string, delTags []string) error {
	args := []string{"-v", name}
	for _, tag := range addTags {
		if err := ValidateTag(tag); err != nil {
			return err
		}
		args = append(args, "--addtag", tag)
	}
	for _, tag := range delTags {
		if err := ValidateTag(tag); err != nil {
			return err
		}
		args = append(args, "--deltag", tag)
	}
	if _, err := runLvm(ctx, cmd, args...); err != nil {
		return err
	}
	return nil
}

func ChangeTagsPV(ctx context.Context, name string, addTags []string, delTags []string) error {
	return doChangeTags(ctx, "pvchange", name, addTags, delTags)
}

func ChangeTagsVG(ctx context.Context, name string, addTags []string, delTags []string) error {
	return doChangeTags(ctx, "vgchange", name, addTags, delTags)
}

func ChangeTagsLV(ctx context.Context, vg string, name string, addTags []string, delTags []string) error {
	return doChangeTags(ctx, "lvchange", volumePath(vg, name), addTags, delTags)
}

func runLvm(ctx context.Context, cmd string, args ...string) (string, error) {
	sp, _ := opentracing.StartSpanFromContext(ctx, "lvm/"+cmd, lvmTag, ext.SpanKindRPCClient)

	defer sp.Finish()

	c := exec.Command(cmd, args...)
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr
	err := c.Run()
	errstr := ignoreWarnings(stderr.String())
	stdstr := stdout.String()
	lf := []zap.Field{
		zap.String("cmd", c.String()), zap.Error(err),
		zap.String("stdout", stdstr),
		zap.String("stderr", errstr),
	}
	if err != nil {
		utils.LogSpanError(sp, err)
		ilog.Log().Error("fail", lf...)
		return stdstr, fmt.Errorf("%s faild with stdout:'%s', err:%w", c.String(), errstr, err)
	}
	ilog.Log().Debug("exec", lf...)
	return stdstr, nil
}

func ignoreWarnings(str string) string {
	lines := strings.Split(str, "\n")
	result := make([]string, 0, len(lines))
	for _, line := range lines {
		line = strings.TrimSpace(line)
		// Ignore annoing leaked file descriptors
		// LVM is very suspicious about filedescriptors opened by external app
		// https://github.com/mesosphere/csilvm/blob/ea96ea409c7ff49bdefb8d0e75ca477f3e6c012c/pkg/lvm/lvm.go#L906
		if strings.HasPrefix(line, "WARNING") || strings.HasPrefix(line, "File descriptor") {
			ilog.Log().Debug(line)
			continue
		}
		result = append(result, line)
	}
	return strings.TrimSpace(strings.Join(result, "\n"))
}

func SetupTestLVMconfig(ctx context.Context) error {
	data, err := runLvm(ctx, "lvmconfig", "--mergedconfig", "--config", "devices{filter = [ \"a|.*/|\" ]}")
	if err != nil {
		return err
	}
	return ioutil.WriteFile("/etc/lvm/lvm.conf", []byte(data), 0644)
}
