package lvm

import (
	"context"
	"os/user"
	"testing"

	ot "github.com/opentracing/opentracing-go"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/diskmanager/pkg/loop"
)

type noopSpan struct{}

const (
	devSize = uint64(128 << 20)
	devName = "test.img"
)

var (
	tracer = ot.NoopTracer{}
	span   = tracer.StartSpan("commands_test_main")
)

func requireRoot(t *testing.T) {
	u, _ := user.Current()
	assert.NotNil(t, u)
	if u.Uid != "0" {
		t.Skipf("Test requires root privileges, current uid is: %s ", u.Uid)
	}
}

func TestCreatePhysicalDevice(t *testing.T) {
	requireRoot(t)

	ctx := context.Background()
	loop, err := loop.CreateTMPLoopDevice(devSize, devName)
	if err != nil {
		t.Fatal(err)
	}
	defer loop.Close()

	err = PVScan(ctx, loop.Path())
	assert.NoError(t, err)
	err = CreatePV(ctx, loop.Path())
	assert.NoError(t, err)
	err = RemovePV(ctx, loop.Path())
	assert.NoError(t, err)
}

func TestCreateLV(t *testing.T) {
	requireRoot(t)

	ctx := context.Background()
	err := SetupTestLVMconfig(ctx)
	require.NoError(t, err)

	loop, err := loop.CreateTMPLoopDevice(devSize, devName)
	if err != nil {
		t.Fatal(err)
	}
	defer loop.Close()

	err = CreatePV(ctx, loop.Path())
	require.NoError(t, err)

	vgTags := []string{"tag1", "tag2"}
	lvTags := []string{"tag3", "tag4"}

	err = CreateVG(ctx, loop.Path(), "vgt1", vgTags)
	assert.NoError(t, err)

	err = CreateLV(ctx, "vgt1", "lvt1", 32<<20, lvTags)
	assert.NoError(t, err)

	vg, err := LookupVG(ctx, "vgt1")
	assert.NoError(t, err)
	assert.Equal(t, vgTags, vg.Tags)
	assert.Equal(t, uint64(1), vg.LvCount)

	pv, err := LookupPV(ctx, loop.Path())
	assert.NoError(t, err)
	assert.Equal(t, vg.UUID, pv.VgUUID)

	lv, err := LookupLV(ctx, "vgt1", "lvt1")
	assert.NoError(t, err)
	assert.Equal(t, lvTags, lv.Tags)
	assert.Equal(t, vg.UUID, lv.VgUUID)
	assert.FileExists(t, lv.DevPath)

	ilog.Log().Info("dump info", zap.Any("pv", pv), zap.Any("vg", vg), zap.Any("lv", lv))
	err = RemoveLV(ctx, "vgt1", "lvt1")
	assert.NoError(t, err)

	err = RemoveVG(ctx, "vgt1")
	assert.NoError(t, err)

	err = RemovePV(ctx, loop.Path())
	assert.NoError(t, err)
}
