package lvm

import (
	"fmt"
	"path/filepath"
	"strconv"
	"strings"
)

const separator = "<:XSEP:>"

// VolumePermissions is volume permissions
type VolumePermissions rune

var volumePermissonsKeys = []byte("wrR")

// permissions
const (
	VolumePermissionsReadOnly           VolumePermissions = 'r'
	VolumePermissionsReadOnlyActivation VolumePermissions = 'R'
	VolumePermissionsWriteable          VolumePermissions = 'w'
)

// VolumeAllocation is volume allocation policy
type VolumeAllocation rune

var volumeAllocationKeys = []byte("acilnACILN")

// allocations
const (
	VolumeAllocationAnywhere         VolumeAllocation = 'a'
	VolumeAllocationContiguous       VolumeAllocation = 'c'
	VolumeAllocationInherited        VolumeAllocation = 'i'
	VolumeAllocationCling            VolumeAllocation = 'l'
	VolumeAllocationNormal           VolumeAllocation = 'n'
	VolumeAllocationAnywhereLocked   VolumeAllocation = 'A'
	VolumeAllocationContiguousLocked VolumeAllocation = 'C'
	VolumeAllocationInheritedLocked  VolumeAllocation = 'I'
	VolumeAllocationClingLocked      VolumeAllocation = 'L'
	VolumeAllocationNormalLocked     VolumeAllocation = 'N'
)

// VolumeState is volume state
type VolumeState rune

var volumeStateKeys = []byte("asISmMdi")

// states
const (
	VolumeStateActive                               VolumeState = 'a'
	VolumeStateSuspended                            VolumeState = 's'
	VolumeStateInvalidSnapshot                      VolumeState = 'I'
	VolumeStateInvalidSuspendedSnapshot             VolumeState = 'S'
	VolumeStateSnapshotMergeFailed                  VolumeState = 'm'
	VolumeStateSuspendedSnapshotMergeFailed         VolumeState = 'M'
	VolumeStateMappedDevicePresentWithoutTables     VolumeState = 'd'
	VolumeStateMappedDevicePresentWithInactiveTable VolumeState = 'i'
)

// VolumeOpen is volume open
type VolumeOpen rune

// open
const (
	VolumeOpenIsOpen    VolumeOpen = 'o'
	VolumeOpenIsNotOpen VolumeOpen = '-'
)

// VolumeHealth is volume health
type VolumeHealth rune

var volumeHealthKeys = []byte("-prmw")

// health
const (
	VolumeHealthOK              VolumeHealth = '-'
	VolumeHealthPartial         VolumeHealth = 'p'
	VolumeHealthRefreshNeeded   VolumeHealth = 'r'
	VolumeHealthMismatchesExist VolumeHealth = 'm'
	VolumeHealthWritemostly     VolumeHealth = 'w'
)

type PhysicalVolume struct {
	Name     string
	Size     uint64
	FreeSize uint64
	UUID     string
	VgUUID   string
	Tags     []string
}

type VolumeGroup struct {
	Name     string
	Size     uint64
	FreeSize uint64
	UUID     string
	SysID    string
	LvCount  uint64
	Tags     []string
}

// LVAttributes is attributes
type LVAttributes struct {
	Permissions VolumePermissions
	Allocation  VolumeAllocation
	State       VolumeState
	Open        VolumeOpen
	Health      VolumeHealth
}

type LogicalVolume struct {
	Name         string
	Size         uint64
	UUID         string
	VgUUID       string
	DevPath      string
	DevMajNumber uint32
	DevMinNumber uint32
	Attributes   LVAttributes
	Tags         []string
}

func parse(line string, nrTok int) (map[string]string, error) {
	tokens := strings.Split(line, separator)
	if len(tokens) != nrTok {
		return nil, makeLvmParseError("expected %d tokens, got %d", nrTok, len(tokens))
	}

	tok := map[string]string{}
	for _, c := range tokens {
		idx := strings.Index(c, "=")
		if idx == -1 {
			return nil, makeLvmParseError("failed to parse token '%s'", c)
		}
		key := c[0:idx]
		val := c[idx+1:]
		if len(val) < 2 {
			return nil, makeLvmParseError("failed to parse token '%s'", c)
		}
		if val[0] != '\'' || val[len(val)-1] != '\'' {
			return nil, makeLvmParseError("failed to parse token '%s'", c)
		}
		val = val[1 : len(val)-1]
		tok[key] = val
	}

	return tok, nil
}

func parseLVAttrs(attrs string) (*LVAttributes, error) {
	// In case of legacy lvmAPI used, simply set values to default mode
	if len(attrs) == 6 {
		attrs += "----"
	}
	if len(attrs) != 10 {
		return nil, fmt.Errorf("incorrect attrs block size, expected 10, got %d in %s", len(attrs), attrs)
	}
	ret := &LVAttributes{}
	ret.Permissions = VolumePermissions(attrs[1])
	ret.Allocation = VolumeAllocation(attrs[2])
	ret.State = VolumeState(attrs[4])
	ret.Open = VolumeOpen(attrs[5])
	ret.Health = VolumeHealth(attrs[8])

	return ret, nil
}

func parseLV(line string, realPath bool) (*LogicalVolume, error) {
	// 18.06:(lvm2. 2.02.176-4.1ubuntu3.18.04.1):
	// chunk_size, convert_lv, copy_percent, data_lv, devices, discards, lv_attr, lv_host, lv_kernel_major, lv_kernel_minor, lv_kernel_read_ahead, lv_major, lv_minor,  lv_name,
	// lv_path,  lv_profile,  lv_read_ahead,  lv_size,  lv_tags,  lv_time,  lv_uuid,  metadata_lv, mirror_log, modules, move_pv, origin, origin_size, pool_lv, raid_max_recovery_rate, raid_min_recov‐
	// ery_rate, raid_mismatch_count, raid_sync_action, raid_write_behind, region_size, segtype, seg_count, seg_pe_ranges, seg_size, seg_size_pe,  seg_start,  seg_start_pe,  seg_tags,  snap_percent,
	//
	// 12.04(lvm2.2.02.66-4ubuntu7.4):
	// stripes, stripe_size, sync_percent, thin_count, transaction_id, zero, lv_uuid, lv_name, lv_attr, lv_major, lv_minor, lv_read_ahead, lv_kernel_major, lv_kernel_minor, lv_kernel_read_ahead,
	// lv_size, seg_count, origin, origin_size, snap_per‐cent, copy_percent, move_pv, convert_lv, lv_tags, mirror_log, modules, segtype, stripes,  stripesize,  regionsize,  chunksize,  seg_start,
	// seg_start_pe,  seg_size,  seg_tags,  seg_pe_ranges, devices.
	//

	// lvs --units=b --separator=<:XSEP:> --nosuffix --noheadings --nameprefixes -a -o lv_uuid,lv_name,lv_size,lv_attr,lv_kernel_major,lv_kernel_minor,lv_tags,vg_uuid,vg_name
	fields, err := parse(line, 9)
	if err != nil {
		return nil, err
	}

	size, err := strconv.ParseUint(fields["LVM2_LV_SIZE"], 10, 64)
	if err != nil {
		return nil, err
	}

	devMaj, err := strconv.ParseUint(fields["LVM2_LV_KERNEL_MAJOR"], 10, 32)
	if err != nil {
		return nil, err
	}

	devMin, err := strconv.ParseUint(fields["LVM2_LV_KERNEL_MINOR"], 10, 32)
	if err != nil {
		return nil, err
	}

	attrs, err := parseLVAttrs(fields["LVM2_LV_ATTR"])
	if err != nil {
		return nil, err
	}
	name := fields["LVM2_LV_NAME"]
	vg := fields["LVM2_VG_NAME"]
	devPath := fmt.Sprintf("/dev/%s/%s", vg, name)
	if realPath {
		devPath, err = filepath.EvalSymlinks(devPath)
		if err != nil {
			return nil, err
		}
	}
	return &LogicalVolume{
		Name:   name,
		Size:   size,
		UUID:   fields["LVM2_LV_UUID"],
		VgUUID: fields["LVM2_VG_UUID"],
		// TODO replace it with "LVM2_LV_PATH" once we drop precise distro
		DevPath:      devPath,
		DevMajNumber: uint32(devMaj),
		DevMinNumber: uint32(devMin),
		Attributes:   *attrs,
		Tags:         strings.Split(fields["LVM2_LV_TAGS"], ","),
	}, nil
}

func parseVG(line string) (*VolumeGroup, error) {
	// vgs --units=b --separator="<:XSEP:>" --nosuffix --noheadings -o vg_uuid,vg_name,vg_size,vg_free,vg_sysid,vg_extent_size,vg_tags,lv_count --nameprefixes -a
	fields, err := parse(line, 8)
	if err != nil {
		return nil, err
	}

	size, err := strconv.ParseUint(fields["LVM2_VG_SIZE"], 10, 64)
	if err != nil {
		return nil, err
	}

	freeSize, err := strconv.ParseUint(fields["LVM2_VG_FREE"], 10, 64)
	if err != nil {
		return nil, err
	}

	volumes, err := strconv.ParseUint(fields["LVM2_LV_COUNT"], 10, 64)
	if err != nil {
		return nil, err
	}

	return &VolumeGroup{
		Name:     fields["LVM2_VG_NAME"],
		Size:     size,
		FreeSize: freeSize,
		UUID:     fields["LVM2_VG_UUID"],
		SysID:    fields["LVM_VG_SYSID"],
		LvCount:  volumes,
		Tags:     strings.Split(fields["LVM2_VG_TAGS"], ","),
	}, nil
}

func parsePV(line string) (*PhysicalVolume, error) {
	// pvs --units=b --separator="<:XSEP:>" --nosuffix --noheadings -o pv_name,pv_uuid,pv_size,pv_free,pv_tags,pe_start,dev_size,vg_uuid,vg_name --nameprefixes -a
	fields, err := parse(line, 9)
	if err != nil {
		return nil, err
	}

	size, err := strconv.ParseUint(fields["LVM2_PV_SIZE"], 10, 64)
	if err != nil {
		return nil, err
	}

	freeSize, err := strconv.ParseUint(fields["LVM2_PV_FREE"], 10, 64)
	if err != nil {
		return nil, err
	}
	return &PhysicalVolume{
		Name:     fields["LVM2_PV_NAME"],
		Size:     size,
		FreeSize: freeSize,
		UUID:     fields["LVM2_PV_UUID"],
		VgUUID:   fields["LVM2_VG_UUID"],
		Tags:     strings.Split(fields["LVM2_PV_TAGS"], ","),
	}, nil
}
