package lvm

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestParseLVAttrsBad(t *testing.T) {
	attrs, err := parseLVAttrs("bad-bad")
	assert.Error(t, err)
	assert.Nilf(t, attrs, "should fail to parse %v", attrs)
}

func TestParseLVAttrsLegacy(t *testing.T) {
	attrs, err := parseLVAttrs("-wi-a-")
	assert.NoError(t, err)
	assert.NotNil(t, attrs)

	assert.Equal(t, attrs.Permissions, VolumePermissionsWriteable)
	assert.Equal(t, attrs.Allocation, VolumeAllocationInherited)
	assert.Equal(t, attrs.State, VolumeStateActive)
	assert.Equal(t, attrs.Open, VolumeOpenIsNotOpen)
	assert.Equal(t, attrs.Health, VolumeHealthOK)

	// Compare legacy and modern
	attrs2, err2 := parseLVAttrs("-wi-a-----")
	assert.Equal(t, err, err2)
	assert.Equal(t, attrs, attrs2)
}

func TestParseLVAttrs(t *testing.T) {
	attrs, err := parseLVAttrs("-wi-ao----")
	assert.NoError(t, err)
	assert.NotNil(t, attrs)

	assert.Equal(t, attrs.Permissions, VolumePermissionsWriteable)
	assert.Equal(t, attrs.Allocation, VolumeAllocationInherited)
	assert.Equal(t, attrs.State, VolumeStateActive)
	assert.Equal(t, attrs.Open, VolumeOpenIsOpen)
	assert.Equal(t, attrs.Health, VolumeHealthOK)
}

func TestParseLV(t *testing.T) {
	lvSample := "LVM2_LV_UUID='ELWjMT-9ICZ-p2fI-WvpH-cjNU-7jYS-4pS37n'<:XSEP:>LVM2_LV_NAME='test'<:XSEP:>LVM2_LV_SIZE='10737418240'<:XSEP:>LVM2_LV_ATTR='-wi-a-----'<:XSEP:>LVM2_LV_KERNEL_MAJOR='252'<:XSEP:>LVM2_LV_KERNEL_MINOR='3'<:XSEP:>LVM2_LV_TAGS='diskman.access_type=mount,diskman.fs_type=ext4,diskman.fstrim=1,diskman.mount_policy=DEFAULT,diskman=true'<:XSEP:>LVM2_VG_UUID='HXQtIx-ODEX-qndE-71fo-yMAF-EMP2-0xtKox'<:XSEP:>LVM2_VG_NAME='hdd'"
	attrGold := LVAttributes{
		Permissions: VolumePermissionsWriteable,
		Allocation:  VolumeAllocationInherited,
		State:       VolumeStateActive,
		Open:        VolumeOpenIsNotOpen,
		Health:      VolumeHealthOK,
	}
	lvGold := &LogicalVolume{
		Name:         "test",
		Size:         10737418240,
		UUID:         "ELWjMT-9ICZ-p2fI-WvpH-cjNU-7jYS-4pS37n",
		VgUUID:       "HXQtIx-ODEX-qndE-71fo-yMAF-EMP2-0xtKox",
		DevPath:      "/dev/hdd/test",
		DevMajNumber: 252,
		DevMinNumber: 3,
		Attributes:   attrGold,
		Tags:         []string{"diskman.access_type=mount", "diskman.fs_type=ext4", "diskman.fstrim=1", "diskman.mount_policy=DEFAULT", "diskman=true"},
	}
	lv, err := parseLV(lvSample, false)
	assert.NoError(t, err)
	assert.Equal(t, lvGold, lv)
}
