package mountinfo

import (
	"bufio"
	"fmt"
	"io"
	"os"
	"strconv"
	"strings"

	"a.yandex-team.ru/infra/rsm/diskmanager/internal/utils"
	"a.yandex-team.ru/library/go/core/xerrors"
)

func doParseInfoFile(r io.Reader) ([]MountInfo, error) {
	s := bufio.NewScanner(r)
	lst := []MountInfo{}
	var err error
	for s.Scan() {
		if err = s.Err(); err != nil {
			return nil, err
		}

		/*
		  52 24 252:42 /    /place rw,relatime shared:34 - ext4 /dev/mapper/hdd-place rw,quota,usrquota
		  (1)(2)(3)   (4)   (5)      (6)       (7)      (8) (9) (10)                  (11)
		  (1) mount ID:  unique identifier of the mount (may be reused after umount)
		  (2) parent ID:  ID of parent (or of self for the top of the mount tree)
		  (3) major:minor:  value of st_dev for files on filesystem
		  (4) root:  root of the mount within the filesystem
		  (5) mount point:  mount point relative to the process's root
		  (6) mount options:  per mount options
		  (7) optional fields:  zero or more fields of the form "tag[:value]"
		  (8) separator:  marks the end of the optional fields
		  (9) filesystem type:  name of filesystem of the form "type[.subtype]"
		  (10) mount source:  filesystem specific information or "none"
		  (11) super options:  per super block options
		*/

		text := s.Text()
		fields := strings.Split(text, " ")
		numFields := len(fields)
		if numFields < 10 {
			// should be at least 10 fields
			return nil, fmt.Errorf("'%s' failed: not enough fields (%d), 10", text, numFields)
		}
		m := MountInfo{}
		m.ID, err = utils.ParseUint32(fields[0])
		if err != nil {
			return nil, err
		}
		m.Parent, err = utils.ParseUint32(fields[1])
		if err != nil {
			return nil, err
		}
		m.MajorMinor = fields[2]
		mm := strings.Split(m.MajorMinor, ":")
		if len(mm) != 2 {
			return nil, fmt.Errorf("'%s' failed: unexpected minor:major pair %s", text, mm)
		}
		m.Major, err = utils.ParseUint32(mm[0])
		if err != nil {
			return nil, err
		}
		m.Minor, err = utils.ParseUint32(mm[1])
		if err != nil {
			return nil, err
		}

		/* 4 */
		m.Root, err = strconv.Unquote(`"` + fields[3] + `"`)
		if err != nil {
			return nil, fmt.Errorf("'%s' failed: unable to unquote root field err: %w", fields[3], err)
		}
		m.Mountpoint, err = strconv.Unquote(`"` + fields[4] + `"`)
		if err != nil {
			return nil, fmt.Errorf("'%s' failed: unable to unquote mount point field, err:%w", fields[4], err)
		}
		m.Options = strings.Split(fields[5], ",")

		// one or more optional fields, when a separator (-)
		i := 6
		for ; i < numFields && fields[i] != "-"; i++ {
			switch i {
			case 6:
				/* TODO Fill Tags m.Tags = fields[6] */
			default:
				/* NOTE there might be more optional fields before the separator
				   such as fields[7]...fields[N] (where N < separatorIndex),
				   although as of Linux kernel 4.15 the only known ones are
				   mount propagation flags in fields[6]. The correct
				   behavior is to ignore any unknown optional fields.
				*/
			}
		}
		if i == numFields {
			return nil, fmt.Errorf("'%s' failed: missing separator ('-')", text)
		}
		/* 7'th field */
		if i+4 > numFields {
			return nil, fmt.Errorf("'%s' failed: not enough fields after a separator", text)
		}
		// From https://github.com/containerd/containerd/blob/master/mount/mountinfo_linux.go#L120
		// Parse only known ones, ignore all others
		m.FsType = fields[i+1]
		m.Source = fields[i+2]
		m.FsOptions = strings.Split(fields[i+3], ",")
		lst = append(lst, m)
	}
	return lst, nil
}
func parseInfoFile(r io.Reader) ([]MountInfo, error) {
	ret, err := doParseInfoFile(r)
	if err != nil {
		return ret, xerrors.Errorf("mountInfo parse: %w", err)
	}
	return ret, nil
}

// Self retrieves a list of mounts for the current running process.
func Self() ([]MountInfo, error) {
	f, err := os.Open("/proc/self/mountinfo")
	if err != nil {
		return nil, err
	}
	defer f.Close()

	return parseInfoFile(f)
}

// Retrives mouninfo list for given PID
func PID(pid int) ([]MountInfo, error) {
	f, err := os.Open(fmt.Sprintf("/proc/%d/mountinfo", pid))
	if err != nil {
		return nil, err
	}
	defer f.Close()

	return parseInfoFile(f)
}
