package sysfs

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"strconv"
	"strings"
)

type Attribute string

func (attrib Attribute) Exists() bool {
	info, err := os.Stat(string(attrib))
	return err == nil && !info.IsDir()
}

func (attrib Attribute) String() string {
	return string(attrib)
}

func (attrib Attribute) Attribute(name string) Attribute {
	return Attribute(string(attrib) + "/" + name)
}

func (attrib Attribute) Read() (string, error) {
	data, err := ioutil.ReadFile(string(attrib))
	if err != nil {
		return "", err
	}
	return strings.TrimSpace(string(data)), nil
}

func (attrib Attribute) Readlink() (string, error) {
	return os.Readlink(string(attrib))
}

func (attrib Attribute) ReadAbslink() (string, error) {
	path, err := attrib.Readlink()
	if err != nil {
		return "", err
	}
	return filepath.Abs(string(attrib) + "/" + path)
}

func (attrib Attribute) Write(value string) error {
	return ioutil.WriteFile(string(attrib), []byte(value), 0666)
}

func (attrib Attribute) ReadInt() (int, error) {
	s, err := attrib.Read()
	if err != nil {
		return 0, err
	}
	return strconv.Atoi(s)
}

func (attrib Attribute) ReadBool() (bool, error) {
	s, err := attrib.Read()
	if err != nil {
		return false, err
	}
	v, err := strconv.Atoi(s)
	return !(v == 0), err
}

func (attrib Attribute) ReadUint64() (value uint64, err error) {
	s, err := attrib.Read()
	if err != nil {
		return 0, err
	}

	return strconv.ParseUint(s, 10, 64)
}

func (attrib Attribute) WriteInt(value int) (err error) {
	return attrib.Write(strconv.Itoa(value))
}

func (attrib Attribute) WriteUint64(value uint64) (err error) {
	return attrib.Write(strconv.FormatUint(value, 10))
}
