package sysfs

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
)

type Object string

func (obj Object) Name() string {
	return string(obj)[strings.LastIndex(string(obj), "/")+1:]
}

func (obj Object) String() string {
	return string(obj)
}

func (obj Object) Exists() bool {
	info, err := os.Stat(string(obj))
	return err == nil && info.IsDir()
}

func (obj Object) Readlink() (string, error) {
	return os.Readlink(string(obj))
}

func (obj Object) ReadAbslink() (string, error) {
	path, err := obj.Readlink()
	if err != nil {
		return "", err
	}
	return filepath.Abs(string(obj) + "/" + path)
}

func (obj Object) GetSubObjects() ([]Object, error) {
	files, err := ioutil.ReadDir(string(obj))
	if err != nil {
		return nil, err
	}
	path := string(obj) + "/"
	objects := make([]Object, len(files))
	for _, info := range files {
		objects = append(objects, Object(path+info.Name()))
	}
	return objects, nil
}

func NewObject(name string) Object {
	return Object(name)
}

func (obj Object) SubObject(name string) Object {
	return Object(string(obj) + "/" + name)
}

func (obj Object) GetAttributes() ([]Attribute, error) {
	files, err := ioutil.ReadDir(string(obj))
	if err != nil {
		return nil, err
	}
	path := string(obj) + "/"
	attrs := make([]Attribute, 0)
	for _, info := range files {
		if !info.IsDir() {
			attrs = append(attrs, Attribute(path+info.Name()))
		}
	}
	return attrs, nil
}

func (obj Object) Attribute(name string) Attribute {
	return Attribute(string(obj) + "/" + name)
}
