package udev

import (
	"bufio"
	"fmt"
	"io"
	"os"
	"strconv"
	"strings"
)

// parseFile parse udev data file
func parseFile(r io.Reader) (*Entry, error) {
	s := bufio.NewScanner(r)
	e := Entry{}
	e.Links = make([]string, 1)
	e.Env = map[string]string{}

	var err error

	for s.Scan() {
		if err := s.Err(); err != nil {
			return nil, err
		}
		/* TOKEN_TYPE:TOKEN
		                   Example:
				   S:disk/by-id/ata-Hitachi_HUA722010CLA330_JPW9H0N018KR8V
				   W:105212
				   I:7506308
				   E:ID_MODEL_ENC=Hitachi\x20HUA722010CLA330\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20
				   E:ID_REVISION=JP4OA3EA
		*/

		text := s.Text()
		fields := strings.SplitN(text, ":", 2)
		if len(fields) != 2 {
			return nil, fmt.Errorf("'%s'Bad number of fields want 2, got:%v", text, fields)
		}
		tokType := fields[0]
		tokVal := fields[1]
		switch tokType {
		case "S":
			e.Links = append(e.Links, tokVal)
		case "E":
			kv := strings.SplitN(tokVal, "=", 2)
			if len(kv) != 2 {
				// TODO, may be we can simply skip broken entry
				return nil, fmt.Errorf("'%s'Bad number of fields want 2, got:%v", text, fields)
			}
			if len(kv[1]) != 0 {
				e.Env[kv[0]] = kv[1]
			}
		case "I":
			e.InitTS, err = strconv.ParseUint(tokVal, 10, 64)
			if err != nil {
				return nil, fmt.Errorf("'%s' Bad init timestamp ", text)
			}
		default:
			// TODO Add debug log here
		}
	}
	return &e, nil
}

// BlockDev entry Reader
func (d BlockDev) Read() (*Entry, error) {
	f, err := os.Open(UdevRoot + "/b" + string(d))
	if err != nil {
		return nil, err
	}
	defer f.Close()

	e, err := parseFile(f)
	if err != nil {
		return nil, err
	}
	for idx, link := range e.Links {
		e.Links[idx] = "/dev/disk/" + link
	}
	return e, nil
}
