package cmd

import (
	"os"
	"path/filepath"
	"syscall"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"

	"github.com/spf13/cobra"
)

const (
	defaultCmd      = "coredns"
	defaultConfDir  = "/etc/yandex-dns-manager"
	defaultLogDir   = "/var/log/yandex-dns-manager"
	defaultLogName  = "dnsmanager.log"
	logrotateSignal = syscall.SIGHUP
)

var (
	confDir  string
	logDir   string
	debug    bool
	logLevel = log.InfoLevel
	logger   *zap.Logger
)

func Execute() {
	rootCmd := &cobra.Command{
		Use:   "dnsmanager",
		Short: "DNS manager",
		Long: `DNS manager,
      Wiki: https://wiki.yandex-team.ru/rtcnetdev/dns/hostdnscache/#dnsmanager
      Docs: https://a.yandex-team.ru/arc/trunk/arcadia/infra/dnsmanager/README.md`,
		Version: buildinfo.Info.ProgramVersion,
	}

	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}

	rootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "enable debug level")
	rootCmd.PersistentFlags().StringVar(&confDir, "conf_dir", defaultConfDir, "path to configuration files")
	rootCmd.PersistentFlags().StringVar(&logDir, "log_dir", defaultLogDir, "path to log files")

	registerCoreDNS(rootCmd)
	registerCheck(rootCmd)

	_, _, err := rootCmd.Find(os.Args[1:])
	if err != nil || len(os.Args) == 1 {
		rootCmd.SetArgs([]string{defaultCmd})
	}

	if err := rootCmd.Execute(); err != nil {
		panic(err)
	}
}

func setupLogger() error {
	logPath := filepath.Join(logDir, defaultLogName)
	err := logrotate.RegisterLogrotateSink(logrotateSignal)
	if err != nil {
		return err
	}
	if debug {
		logLevel = log.DebugLevel
	}
	cfg := zap.ConsoleConfig(logLevel)
	// it'l rotate by signal
	cfg.OutputPaths = []string{"logrotate://" + logPath}
	cfg.ErrorOutputPaths = []string{logPath}
	cfg.DisableStacktrace = true
	logger, err = zap.New(cfg)
	if err != nil {
		return err
	}
	return nil
}
