package libcoredns

import (
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"syscall"

	"a.yandex-team.ru/infra/rsm/dnsmanager/pkg/serverinfo"
	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/resource"

	"github.com/coredns/caddy"
	"github.com/coredns/coredns/core/dnsserver"
	gozap "go.uber.org/zap"

	// Plugins
	_ "a.yandex-team.ru/infra/rsm/dnsmanager/internal/libcoredns/plugin/dummy"
	_ "a.yandex-team.ru/infra/rsm/dnsmanager/internal/libcoredns/plugin/staticfile"
	_ "a.yandex-team.ru/infra/rsm/dnsmanager/internal/libcoredns/plugin/yasm"

	_ "github.com/coredns/coredns/plugin/bind"
	_ "github.com/coredns/coredns/plugin/cache"
	_ "github.com/coredns/coredns/plugin/root"
	//_ "github.com/coredns/coredns/plugin/errors"
	_ "github.com/coredns/coredns/plugin/file"
	_ "github.com/coredns/coredns/plugin/forward"
	_ "github.com/coredns/coredns/plugin/loadbalance"
	_ "github.com/coredns/coredns/plugin/log"
	_ "github.com/coredns/coredns/plugin/metrics"
	_ "github.com/coredns/coredns/plugin/rewrite"
)

const (
	AppName         = "yandex-dns-manager"
	ServerType      = "dns"
	SubdirConfPath  = "libcoredns"
	DefaultConfName = "coredns.conf"
)

var (
	PAGESIZE = os.Getpagesize()
)

type coredns struct {
	l       *zap.Logger
	confDir string
}

func NewCoreDNS(l *zap.Logger, d string) *coredns {
	named := &coredns{
		l:       l,
		confDir: filepath.Join(d, SubdirConfPath),
	}
	// coredns uses system logger "log" and writes messages to stdout
	// write stdout/err to zap logger
	gozap.RedirectStdLog(l.L)

	caddy.AppName = AppName
	caddy.AppVersion = buildinfo.Info.SVNRevision
	// Quiet mode will not show any informative output on initialization
	caddy.Quiet = true
	dnsserver.Quiet = true
	// Reads configs in order which LoaderFunc's are registered.
	// LoaderFunc must return error nil if needs to try next loader,
	// otherwise coredns will immediately quit with error
	caddy.RegisterCaddyfileLoader("fs", caddy.LoaderFunc(named.fsConfLoader))
	caddy.RegisterCaddyfileLoader("location", caddy.LoaderFunc(named.locationConfLoader))
	caddy.SetDefaultCaddyfileLoader("default", caddy.LoaderFunc(named.defaultConfLoader))

	return named
}

// Reading custom config (antifuckup capability)
func (c *coredns) fsConfLoader(serverType string) (caddy.Input, error) {
	confPath := filepath.Join(c.confDir, DefaultConfName)
	contents, err := ioutil.ReadFile(confPath)
	if err != nil {
		if os.IsNotExist(err) {
			c.l.Debugf("%s", err)
			return nil, nil
		}
		c.l.Errorf("%s", err)
		return nil, nil
	}
	return caddy.CaddyfileInput{
		Contents:       contents,
		Filepath:       confPath,
		ServerTypeName: serverType,
	}, nil
}

// Reading predefined config based on server location
func (c *coredns) locationConfLoader(serverType string) (caddy.Input, error) {
	si, err := serverinfo.NewDefault().Read()
	if err != nil {
		c.l.Errorf("failed to read server info: %s", err)
		return nil, nil
	}
	confPath := fmt.Sprintf("%s/coredns_%s.conf", SubdirConfPath, si.Location)

	c.l.Debugf("trying getting location based conf from memory: %s", confPath)
	contents := resource.Get(confPath)
	if contents == nil {
		c.l.Errorf("%s file doesn't exist", confPath)
		return nil, nil
	}

	return caddy.CaddyfileInput{
		Contents:       contents,
		Filepath:       confPath,
		ServerTypeName: serverType,
	}, nil
}

// Reading predefined config based on server location
func (c *coredns) defaultConfLoader(serverType string) (caddy.Input, error) {
	confPath := filepath.Join(SubdirConfPath, DefaultConfName)

	c.l.Debugf("trying getting default conf from memory: %s", confPath)
	contents := resource.Get(confPath)
	if contents == nil {
		return nil, fmt.Errorf("%s file doesn't exist", confPath)
	}

	return caddy.CaddyfileInput{
		Contents:       contents,
		Filepath:       confPath,
		ServerTypeName: serverType,
	}, nil
}

func (c *coredns) Run() {
	c.l.Debugf("loaded plugins:\n%s", caddy.DescribePlugins())

	corefile, err := caddy.LoadCaddyfile(ServerType)
	if err != nil {
		c.l.Fatalf("%s", err)
	}
	c.l.Infof("loaded config:\n%s", corefile.Body())

	if err := mlock(); err != nil {
		c.l.Fatalf("failure on mlock: %s", err)
	}

	caddy.TrapSignals()

	instance, err := caddy.Start(corefile)
	if err != nil {
		c.l.Fatalf("%s", err)
	}

	instance.Wait()
}

func mlock() error {
	f, err := os.Open("/proc/self/exe")
	if err != nil {
		return err
	}
	// closing the file descriptor does not unmap the region, see (man 2 mmap)
	defer func() {
		_ = f.Close()
	}()

	stat, err := f.Stat()
	if err != nil {
		return err
	}
	relSize := int(stat.Size())
	// align by page size
	algSize := (relSize + PAGESIZE - 1) &^ (PAGESIZE - 1)
	data, err := syscall.Mmap(int(f.Fd()), 0, algSize, syscall.PROT_READ, syscall.MAP_SHARED)
	if err != nil {
		return err
	}
	err = syscall.Mlock(data)
	if err != nil {
		_ = syscall.Munmap(data)
		return err
	}
	return nil
}
