// arcadia/library/go/yandex/unistat doesn't support push schema.
// Decided to write own simple implementation.
// In future need to use arcadia/library/go/core/metrics/solomon.
// More about push schema: https://wiki.yandex-team.ru/golovan/userdocs/push-api/?from=%2Fgolovan%2Fpush-api%2F
package yasm

import (
	"encoding/json"
	"math"

	dto "github.com/prometheus/client_model/go"
)

type Metrics map[string]Metric

func (ms Metrics) MarshalJSON() ([]byte, error) {
	var l []Metric
	for _, v := range ms {
		l = append(l, v)
	}
	return json.Marshal(l)
}

type Metric interface {
	UpdateFrom(*dto.Metric)
	MarshalJSON() ([]byte, error)
}

type Counter struct {
	// Metric name, without aggregation suffix
	name string
	// Aggregation suffix
	// See more https://wiki.yandex-team.ru/golovan/userdocs/aggregation-types/?from=%2Fgolovan%2Faggregation-types%2F
	sfx string
	// Variable for storing current (last) value
	cv float64
	// Variable for storing previous value
	// Needs for calculating diff between pushes to yasm
	lv float64
}

func NewCounter(n string) Metric {
	return &Counter{
		name: n,
		sfx:  "_tmmm",
	}
}

func (c *Counter) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"name": c.name + c.sfx,
		"val":  c.GetVal(),
	})
}

func (c *Counter) UpdateFrom(m *dto.Metric) {
	c.cv = m.GetCounter().GetValue()
}

func (c *Counter) GetVal() (val float64) {
	if c.cv > c.lv {
		val = c.cv - c.lv
	}
	c.lv = c.cv
	return
}

type Gauge struct {
	name string
	sfx  string
	cv   float64
	lv   float64
}

func NewGauge(n string) Metric {
	return &Gauge{
		name: n,
		sfx:  "_tmmm",
	}
}

func (g *Gauge) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"name": g.name + g.sfx,
		"val":  g.GetVal(),
	})
}

func (g *Gauge) UpdateFrom(m *dto.Metric) {
	g.cv = m.GetGauge().GetValue()
}

func (g *Gauge) GetVal() (val float64) {
	val = g.lv
	g.lv = g.cv
	return
}

type Histogram struct {
	name string
	sfx  string
	cv   map[float64]float64
	lv   map[float64]float64
}

func NewHistogram(n string) Metric {
	return &Histogram{
		name: n,
		sfx:  "_hgram",
		cv:   make(map[float64]float64),
		lv:   make(map[float64]float64),
	}
}

func (h *Histogram) MarshalJSON() ([]byte, error) {
	return json.Marshal(map[string]interface{}{
		"name": h.name + h.sfx,
		"val":  h.GetVal(),
	})
}

func (h *Histogram) UpdateFrom(m *dto.Metric) {
	var prevCount uint64
	for _, b := range m.GetHistogram().Bucket {
		bName := b.GetUpperBound()
		if math.IsInf(bName, 0) {
			continue
		}
		count := b.GetCumulativeCount()
		h.cv[bName] = float64(count - prevCount)
		prevCount = count
	}
}

func (h *Histogram) GetVal() (val []float64) {
	for i, bcv := range h.cv {
		if blv := h.lv[i]; bcv > blv {
			val = append(val, bcv-blv)
		} else {
			val = append(val, 0)
		}
	}
	h.lv = h.cv
	return
}
