package yasm

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"github.com/prometheus/client_golang/prometheus"
	dto "github.com/prometheus/client_model/go"
)

const (
	defaultInterval = 5 * time.Second
	defaultTimeout  = 3 * time.Second
)

var (
	typeToMetricFunc = map[dto.MetricType]func(string) Metric{
		0: NewCounter,
		1: NewGauge,
		4: NewHistogram,
	}
)

type Tags map[string]string

type Yasm struct {
	URL       string  `json:"-"`
	Tags      Tags    `json:"tags"`
	Ms        Metrics `json:"values"`
	gatherer  prometheus.Gatherer
	allowedMs map[string]map[string]bool
	nReplaces []string
	lReplaces []string
	prfx      string
	interval  time.Duration
	c         *http.Client
	stop      context.CancelFunc
}

func New(u string, t Tags) *Yasm {
	return &Yasm{
		URL:       u,
		Tags:      t,
		Ms:        make(Metrics),
		gatherer:  prometheus.DefaultGatherer,
		allowedMs: make(map[string]map[string]bool),
		nReplaces: []string{},
		lReplaces: []string{},
		interval:  defaultInterval,
		c:         &http.Client{Timeout: defaultTimeout},
	}
}

func (y *Yasm) AddNameReplace(rs []string) {
	y.nReplaces = append(y.nReplaces, rs...)
}

func (y *Yasm) AddLabelReplace(rs []string) {
	y.lReplaces = append(y.lReplaces, rs...)
}

func (y *Yasm) AddAllowedMetric(n string, lns []string) {
	y.allowedMs[n] = map[string]bool{}
	for _, ln := range lns {
		y.allowedMs[n][ln] = true
	}
}

func (y *Yasm) isAllowedMetric(n string, m *dto.Metric) bool {
	l, ok := y.allowedMs[n]
	if !ok {
		return false
	}
	if len(l) == 0 {
		return true
	}
	for _, i := range m.GetLabel() {
		if _, ok := l[i.GetValue()]; ok {
			return true
		}
	}
	return false
}

func (y *Yasm) GetMetric(n string, t dto.MetricType) (Metric, error) {
	if m, ok := y.Ms[n]; ok {
		return m, nil
	}
	f, ok := typeToMetricFunc[t]
	if !ok {
		return nil, fmt.Errorf("unsupported metric type %d", t)
	}
	m := f(n)
	y.Ms[n] = m
	return m, nil

}

func (y *Yasm) OnStartup() error {
	ctx, cancel := context.WithCancel(context.Background())
	y.stop = cancel

	go y.run(ctx)

	return nil
}

func (y *Yasm) OnShutdown() error {
	y.stop()
	return nil
}

func (y *Yasm) run(ctx context.Context) {
	t := time.NewTicker(y.interval)
	defer t.Stop()
	for {
		select {
		case <-ctx.Done():
			return
		case <-t.C:
			if err := y.updateMetrics(); err != nil {
				log.Errorf("error getting metrics: %s", err)
			}
			if err := y.push(ctx); err != nil {
				log.Errorf("error pushing to %s: %s", y.URL, err)
			}
		}
	}
}

func (y *Yasm) push(ctx context.Context) error {
	data, err := json.Marshal([]*Yasm{y})
	if err != nil {
		return err
	}

	log.Debug("push metrics:", string(data))
	req, err := http.NewRequestWithContext(ctx, "POST", "http://"+y.URL, bytes.NewBuffer(data))
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")
	resp, err := y.c.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		body, _ := ioutil.ReadAll(resp.Body)
		return fmt.Errorf("http code %d %s", resp.StatusCode, string(body))
	}
	return nil
}

func (y *Yasm) updateMetrics() error {
	mfs, err := y.gatherer.Gather()
	if err != nil {
		return err
	}
	for _, mf := range mfs {
		for _, m := range mf.GetMetric() {
			// filter only allowed labels
			if !y.isAllowedMetric(mf.GetName(), m) {
				continue
			}
			// concatenate labels and metric name
			name := y.makeMetricName(mf.GetName(), m.GetLabel())
			metric, err := y.GetMetric(name, mf.GetType())
			if err != nil {
				return err
			}
			metric.UpdateFrom(m)
		}
	}
	return nil
}

func (y *Yasm) makeMetricName(n string, mls []*dto.LabelPair) string {
	nr := strings.NewReplacer(y.nReplaces...)
	lr := strings.NewReplacer(y.lReplaces...)
	name := strings.Builder{}
	name.WriteString(y.prfx)
	name.WriteString(n)
	for _, l := range mls {
		// name replacing and some replacing for
		// yasm compability (see https://wiki.yandex-team.ru/golovan/userdocs/tagsandsignalnaming/?from=%2Fgolovan%2Ftagsandsignalnaming%2F)
		lname := lr.Replace(strings.Trim(l.GetValue(), "."))
		if lname != "" {
			name.WriteString("_" + lname)
		}
	}
	// name replacing
	return nr.Replace(name.String())
}
