package named

import (
	"bytes"
	"io/ioutil"
	"os"
	"path"
	"text/template"

	"a.yandex-team.ru/infra/rsm/dnsmanager/pkg/atomicfile"
	"a.yandex-team.ru/library/go/core/resource"
)

const perm os.FileMode = 0644

type TemplatedConf struct {
	tmpl *template.Template
	af   *atomicfile.File
	Data interface{}
}

func (c *TemplatedConf) Update(data interface{}) error {
	var tmplBytes bytes.Buffer
	if data == nil {
		data = c.Data
	}
	if err := c.tmpl.Execute(&tmplBytes, data); err != nil {
		return err
	}
	_, err := tmplBytes.WriteTo(c.af)
	return err
}

func NewTemplatedConf(tmplStr string, dstPath string) (*TemplatedConf, error) {
	tmpl, err := template.New(dstPath).Parse(tmplStr)
	if err != nil {
		return nil, err
	}
	return &TemplatedConf{
		tmpl: tmpl,
		af:   atomicfile.NewFile(dstPath, perm),
	}, nil
}

type Conf struct {
	af   *atomicfile.File
	d    []byte
	path string
}

func NewConf(confDir, srcPath, dstPath string) (*Conf, error) {
	d, err := ReadConf(confDir, srcPath)
	if err != nil {
		return nil, err
	}

	return &Conf{
		af:   atomicfile.NewFile(dstPath, perm),
		d:    d,
		path: dstPath,
	}, nil
}

func (c *Conf) Set() (int, error) {
	return c.af.Write(c.d)
}

func (c *Conf) Path() string {
	return c.path
}

func ReadConf(confDir, confPath string) ([]byte, error) {
	origData := resource.Get(confPath)
	if origData == nil {
		return nil, os.ErrNotExist
	}
	newData, err := ioutil.ReadFile(path.Join(confDir, confPath))
	if err != nil {
		return origData, nil
	}
	return newData, err
}
