package named

import (
	"fmt"
	"io/ioutil"
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestConf(t *testing.T) {
	var (
		confDir = ""
		dstPath = "test.conf"
		tests   = []struct {
			srcPath    string
			data       []byte
			needCreate bool
			err        error
		}{
			{"test1.conf", []byte("test1"), true, nil},
			{"test2.conf", []byte("test2"), true, os.ErrNotExist},
		}
	)
	for _, test := range tests {
		// init conf file
		if test.needCreate {
			err := ioutil.WriteFile(test.srcPath, test.data, perm)
			assert.NoError(t, err)
			if err != nil {
				continue
			}
		}
		conf, err := NewConf(confDir, test.srcPath, dstPath)
		assert.Equal(t, test.err, err)
		if err != nil {
			continue
		}
		// check set
		n, err := conf.Set()
		assert.NoError(t, err)
		assert.NotEqual(t, n, 0)
		// check created file
		data, err := ioutil.ReadFile(dstPath)
		assert.NoError(t, err)
		assert.Equal(t, test.data, data)
		// check perm
		fi, err := os.Stat(dstPath)
		assert.NoError(t, err)
		if err != nil {
			continue
		}
		assert.Equal(t, fi.Mode(), perm)
	}
}

func TestTemplatedConf(t *testing.T) {
	var (
		confPath = "test_config.conf"
		tests    = []struct {
			tmpl     string
			ips      []string
			expected string
		}{
			{
				"acl \"bb_v4\" {\n{{range $ip := .}}{{$ip}};\n{{end}}};\n",
				[]string{"1.1.1.1", "2.2.2.2"},
				"acl \"bb_v4\" {\n1.1.1.1;\n2.2.2.2;\n};\n",
			},
		}
	)
	for _, test := range tests {
		conf, err := NewTemplatedConf(test.tmpl, confPath)
		assert.NoError(t, err)
		err = conf.Update(test.ips)
		assert.NoError(t, err)
		assert.FileExists(t, confPath)
		data, err := ioutil.ReadFile(confPath)
		fmt.Println(data)
		assert.NoError(t, err)
		assert.Equal(t, []byte(test.expected), data)
	}
}
