package atomicfile

import (
	"bufio"
	"bytes"
	"io/ioutil"
	"os"
)

type File struct {
	path    string
	pathTmp string
	perm    os.FileMode
}

func NewFile(p string, perm os.FileMode) *File {
	return &File{
		path:    p,
		pathTmp: p + ".tmp",
		perm:    perm,
	}
}

func (af *File) Write(d []byte) (n int, err error) {
	mod, err := af.ifModified(d)
	if err != nil {
		return
	}
	if !mod {
		return len(d), nil
	}

	f, err := os.Create(af.pathTmp)
	if err != nil {
		return
	}
	defer func() {
		if err != nil {
			_ = f.Close()
			_ = os.Remove(af.pathTmp)
		}
	}()

	bufw := bufio.NewWriter(f)
	n, err = bufw.Write(d)
	if err != nil {
		return
	}
	if err = bufw.Flush(); err != nil {
		return
	}
	if err = f.Chmod(af.perm); err != nil {
		return
	}
	if err = f.Sync(); err != nil {
		return
	}
	if err = os.Rename(af.pathTmp, af.path); err != nil {
		return
	}
	return
}

func (af *File) ifModified(d []byte) (bool, error) {
	newData, err := ioutil.ReadFile(af.path)
	if os.IsNotExist(err) {
		return true, nil
	}
	if err != nil {
		return true, err
	}

	return !bytes.Equal(newData, d), nil
}
