package iface

import (
	"github.com/vishvananda/netlink"
	"net"
	"regexp"
	"syscall"
)

type Iface interface {
	Create() error
	Del() error
	Check() error
}

type Dummy struct {
	Link *netlink.Dummy
	Addr *netlink.Addr
}

func NewDummy(name string, ip string) (*Dummy, error) {
	addr, err := netlink.ParseAddr(ip)
	if err != nil {
		return nil, err
	}

	la := netlink.NewLinkAttrs()
	la.Name = name

	return &Dummy{
		Link: &netlink.Dummy{
			LinkAttrs: la,
		},
		Addr: addr,
	}, nil
}

func (d *Dummy) Create() error {
	if err := netlink.LinkAdd(d.Link); err != nil && err != syscall.EEXIST {
		return err
	}
	if err := netlink.LinkSetUp(d.Link); err != nil {
		return err
	}
	if err := netlink.AddrAdd(d.Link, d.Addr); err != nil && err != syscall.EEXIST {
		return err
	}
	return nil
}

func (d *Dummy) Del() error {
	return nil
}

func (d *Dummy) Check() error {
	return nil
}

func GetUAddrsByIfaceName(rgxp string) ([]string, []string, error) {
	var bbv4IPs, bbv6IPs []string
	ifaces, err := net.Interfaces()
	if err != nil {
		return nil, nil, err
	}
	for _, iface := range ifaces {
		if iface.Flags&net.FlagUp == 0 {
			continue // interface down
		}
		if ok, _ := regexp.MatchString(rgxp, iface.Name); !ok {
			continue
		}
		addrs, err := iface.Addrs()
		if err != nil {
			return nil, nil, err
		}
		for _, addr := range addrs {
			if ipnet, ok := addr.(*net.IPNet); ok && ipnet.IP.IsGlobalUnicast() {
				if ipnet.IP.To4() != nil {
					bbv4IPs = append(bbv4IPs, ipnet.IP.String())
					continue
				} else if ipnet.IP.To16() != nil {
					bbv6IPs = append(bbv6IPs, ipnet.IP.String())
				}
			}
		}
	}
	return bbv4IPs, bbv6IPs, nil
}
