package pushyasm

import (
	"encoding/json"
	"fmt"
	"sync"
)

type Metric interface {
	Inc()
	Set(interface{})
	MarshalJSON() ([]byte, error)
}

type Aggregation string
type AggregationRule int

const (
	Counter Aggregation     = "tmmm"
	Hgram   Aggregation     = "hgram"
	Delta   AggregationRule = iota
	Last
)

type Stats map[string]Metric

func (c *Stats) UnmarshalJSON(data []byte) error {
	var d map[string]float64
	if err := json.Unmarshal(data, &d); err != nil {
		return err
	}
	for k, v := range d {
		if i := (*c)[k]; i != nil {
			i.Set(v)
		}
	}
	return nil
}

type Numeric struct {
	mu        sync.RWMutex
	name      string
	aggr      Aggregation
	localAggr AggregationRule
	lastvalue float64
	value     float64
}

func NewNumeric(name string, aggr Aggregation, localAggr AggregationRule) *Numeric {
	m := &Numeric{
		name:      name,
		aggr:      aggr,
		localAggr: localAggr,
	}
	Reg.Add(m)
	return m
}

func (n *Numeric) Set(v interface{}) {
	n.mu.Lock()
	defer n.mu.Unlock()

	switch value := v.(type) {
	case float64:
		n.value = value
	default:
	}
}

func (n *Numeric) Add(value float64) {
	n.mu.Lock()
	defer n.mu.Unlock()
	n.value += value
}

func (n *Numeric) Inc() {
	n.mu.Lock()
	defer n.mu.Unlock()
	n.value++
}

func (n *Numeric) GetValue() float64 {
	n.mu.Lock()
	defer n.mu.Unlock()

	var value float64
	switch n.localAggr {
	case Delta:
		if n.value > n.lastvalue {
			value = n.value - n.lastvalue
		}
		n.lastvalue = n.value
	default:
		value = n.value
	}
	return value
}

func (n *Numeric) MarshalJSON() ([]byte, error) {
	name := fmt.Sprintf("%s_%s", n.name, n.aggr)
	return json.Marshal(map[string]interface{}{
		"name": name,
		"val":  n.GetValue(),
	})
}

type Histogram struct {
	mu        sync.RWMutex
	name      string
	aggr      Aggregation
	localAggr AggregationRule
	buckets   []float64
	lastvalue []float64
	value     []float64
}

func NewHistogram(name string, start, factor float64, count uint, aggr Aggregation, localAggr AggregationRule) *Histogram {
	buckets := make([]float64, count)
	for i := range buckets {
		buckets[i] = start
		start *= factor
	}
	lastvalue := make([]float64, count)
	value := make([]float64, count)
	m := &Histogram{
		name:      name,
		aggr:      aggr,
		localAggr: localAggr,
		buckets:   buckets,
		lastvalue: lastvalue,
		value:     value,
	}
	Reg.Add(m)
	return m
}

func (n *Histogram) Set(v interface{}) {
	n.mu.Lock()
	defer n.mu.Unlock()

	switch value := v.(type) {
	case map[float64]float64:
		for i, v := range n.buckets {
			if vv, ok := value[v]; ok {
				n.value[i] = vv
			}
		}
	default:
	}
}

func (n *Histogram) Inc() {
	// Not Implemented
}

func (n *Histogram) GetValue() []float64 {
	n.mu.Lock()
	defer n.mu.Unlock()

	value := []float64{}
	switch n.localAggr {
	case Delta:
		for i, v := range n.value {
			if vv := n.lastvalue[i]; v > vv {
				value = append(value, v-vv)
			} else {
				value = append(value, 0)
			}
			n.lastvalue[i] = v
		}
	default:
		value = n.value
	}
	return value
}

func (n *Histogram) MarshalJSON() ([]byte, error) {
	var value [][]float64
	name := fmt.Sprintf("%s_%s", n.name, n.aggr)
	for i, v := range n.GetValue() {
		value = append(value, []float64{n.buckets[i], v})
	}
	value = append(value, []float64{n.buckets[len(n.buckets)-1] * 2, 0})
	return json.Marshal(map[string]interface{}{
		"name": name,
		"val":  value,
	})
}
