package ut_test

import (
	"bytes"
	"context"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"
	"testing"
	"time"

	"a.yandex-team.ru/library/go/test/yatest"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
)

var (
	pkgPath string
)

func init() {
	pkgPath = yatest.BuildPath("infra/rsm/nvgpumanager/build/ci/pkg-target/yandex-nvgpu-manager.0.1-ci.tar.gz")
	if _, err := os.Stat(pkgPath); err != nil {
		panic("does not exists, path: " + pkgPath)
	}
}

func run(ctx context.Context, cmd string, arg ...string) (string, string, error) {
	c := exec.CommandContext(ctx, cmd, arg...)
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr
	err := c.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", c.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))
	return stdout.String(), stderr.String(), err

}

func TestPkgLifeCycle(t *testing.T) {
	var o, e string
	dpkgOpts := []string{"--install"}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	err := os.MkdirAll("pkg", 0755)
	require.NoError(t, err)

	_, e, err = run(ctx, "tar", "vzxf", pkgPath, "-C", "pkg")
	require.NoError(t, err)
	require.Equal(t, "", e)

	files, err := ioutil.ReadDir("pkg")
	require.NoError(t, err)

	for _, f := range files {
		if strings.HasSuffix(f.Name(), "_amd64.deb") {
			dpkgOpts = append(dpkgOpts, "pkg/"+f.Name())
		}
	}

	_, _, err = run(ctx, "dpkg", dpkgOpts...)
	require.NoError(t, err)
	require.Equal(t, "", e)

	eOut := `+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+
| BUS ID | ID | MODEL | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE |
+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+
+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+
+--------+----+-------+--------+------+-------------+
| BUS ID | ID | MODEL | MEMORY | NUMA | IOMMU GROUP |
+--------+----+-------+--------+------+-------------+
+--------+----+-------+--------+------+-------------+
`
	for tries := 20; tries != 0; tries-- {
		o, e, err = run(ctx, "nvgpuctl", "list")
		if err == nil {
			break
		}
		time.Sleep(time.Second)
	}
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
	require.FileExists(t, "/var/log/yandex-nvgpu-manager.log")

	_, _, err = run(ctx, "dpkg", "--remove", "yandex-nvgpu-manager")
	require.NoError(t, err)
	require.Equal(t, "", e)

}
