package ut_test

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"os/exec"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	pb "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/client"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/device"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/utils"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/modprobe"
	"a.yandex-team.ru/library/go/test/yatest"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/config"
)

var (
	serverPath string
	clientPath string
	pkgPath    string
)

func init() {
	// TODO
	serverPath = yatest.BuildPath("infra/rsm/nvgpumanager/cmd/nvgpu-manager/nvgpu-manager")
	clientPath = yatest.BuildPath("infra/rsm/nvgpumanager/cmd/nvgpuctl/nvgpuctl")
	pkgPath = yatest.SourcePath("infra/rsm/nvgpumanager/build/packages/yandex-nvgpu-manager")

	if _, err := os.Stat(serverPath); err != nil {
		panic("does not exists, path: " + serverPath)
	}
	if _, err := os.Stat(clientPath); err != nil {
		panic("does not exists path:" + clientPath)
	}
}

func startDaemon(ctx context.Context, ep string, lockPath, logFile string, mockMode string) (*exec.Cmd, error) {
	cmd := exec.CommandContext(ctx, serverPath,
		"--debug", "--service", ep)
	if logFile != "" {
		cmd.Args = append(cmd.Args, "--log-file", logFile)
	}
	if lockPath != "" {
		cmd.Args = append(cmd.Args, "--lock-file", lockPath)
	}
	if mockMode != config.MockDisabled {
		cmd.Args = append(cmd.Args, "--mock-mode", mockMode)
	}

	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stdout

	err := cmd.Start()
	if err != nil {
		return nil, err
	}

	// Wait for server to start respond
	for tries := 10; tries != 0; tries-- {
		var cl *client.Client

		time.Sleep(time.Second * 1)
		cl, err = client.NewClient(ilog.Log(), ep)
		if err != nil {
			continue
		}
		if cl.Ping(ctx) {
			return cmd, nil
		}
		cl.Close()
	}
	return nil, err
}

func stopDaemon(c *exec.Cmd) {
	if c != nil {
		_ = c.Process.Kill()
		_ = c.Wait()
	}
}

func run(ctx context.Context, cmd string, arg ...string) (string, string, error) {
	c := exec.CommandContext(ctx, cmd, arg...)
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr
	err := c.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", c.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))
	return stdout.String(), stderr.String(), err

}

func getDaemonConf(name string) (string, string) {
	return "unix:" + name + ".sock", name + ".lock"
}

func stripTokens(s string) []string {
	quote := false
	sb := strings.Builder{}
	ret := []string{}
	for _, ch := range s {
		switch ch {
		case '\'':
			quote = !quote
		case ' ':
			if !quote {
				if sb.Len() != 0 {
					ret = append(ret, sb.String())
					sb.Reset()
				}
			} else {
				sb.WriteRune(ch)
			}

		default:
			sb.WriteRune(ch)
		}
	}

	if sb.Len() != 0 {
		ret = append(ret, sb.String())
	}
	return ret
}

func TestListNoDev(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	fmt.Printf("%s %s\n", t.Name(), "enter")
	daemon, err := startDaemon(ctx, ep, lk, "", "disabled")
	defer stopDaemon(daemon)
	require.NoError(t, err)
	eOut := `+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| BUS ID | ID | MODEL | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
+--------+----+-------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
+--------+----+-------+--------+------+-------------+--------+
| BUS ID | ID | MODEL | MEMORY | NUMA | IOMMU GROUP | ACTIVE |
+--------+----+-------+--------+------+-------------+--------+
+--------+----+-------+--------+------+-------------+--------+
`
	fmt.Printf("%s %s\n", t.Name(), "run list")
	o, e, err := run(ctx, clientPath, "list", "--service", ep)
	fmt.Printf("%s %s\n", t.Name(), "ret list")
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
	require.FileExists(t, "/var/log/yandex-nvgpu-manager.log")

	fi, err := os.Stat(ep[5:])
	require.NoError(t, err)
	require.Equal(t, os.FileMode(0o100000664), fi.Mode())
}

func TestCustomLog(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())
	logFile := t.Name() + ".log"
	daemon, err := startDaemon(ctx, ep, lk, logFile, "disabled")
	defer stopDaemon(daemon)
	require.NoError(t, err)
	require.FileExists(t, logFile)
}

func TestLogReopen(t *testing.T) {
	logName := "/var/log/yandex-nvgpu-manager.log"
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", "disabled")
	defer stopDaemon(daemon)
	require.NoError(t, err)
	require.FileExists(t, logName)

	cl, err := client.NewClient(ilog.Log(), ep)
	require.NoError(t, err)
	defer cl.Close()

	err = os.Rename(logName, logName+".old")
	require.NoError(t, err)
	_, _, err = run(ctx, "killall", "-s", "HUP", "nvgpu-manager")
	require.NoError(t, err)

	ok := cl.Ping(ctx)
	require.True(t, ok)
	require.FileExists(t, logName)
}

func TestLogRotate(t *testing.T) {
	logRotCfg := "/etc/logrotate.d/nvgpu-manager"
	logName := "/var/log/yandex-nvgpu-manager.log"

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	o, e, err := run(context.Background(), "groupadd", "--gid", "1334", "diskmanager")
	if err != nil {
		panic(fmt.Sprintf("groupadd failed: stdout: %s, stderr: %s, err: %s", o, e, err))
	}
	err = utils.InstallFile(pkgPath+logRotCfg, logRotCfg, 0644)
	if err != nil {
		panic(fmt.Sprintf("fail to copy:%s err:%v\n", pkgPath+logRotCfg, err))
	}

	daemon, err := startDaemon(ctx, ep, lk, "", "disabled")
	defer stopDaemon(daemon)
	require.NoError(t, err)
	require.FileExists(t, logName)

	cl, err := client.NewClient(ilog.Log(), ep)
	require.NoError(t, err)
	defer cl.Close()

	require.FileExists(t, "/etc/logrotate.d/nvgpu-manager")

	// First log rotate should be noop
	o, e, err = run(ctx, "logrotate", "/etc/logrotate.conf")
	require.NoErrorf(t, err, "logrotate failed stdout:%s, stderr:%s", o, e)
	require.Equal(t, "", o)
	require.Equal(t, "", e)

	require.NoFileExists(t, logName+".1")
	require.NoFileExists(t, logName+".1.gz")

	// Force log size exceed maximum config size of 10mb, should result in log rotate
	err = os.Truncate(logName, 40<<20)
	require.NoError(t, err)

	o, e, err = run(ctx, "logrotate", "/etc/logrotate.conf")
	require.NoErrorf(t, err, "logrotate failed stdout:%s, stderr:%s", o, e)
	require.Equal(t, "", o)
	require.Equal(t, "", e)
	require.FileExists(t, logName+".1.gz")

	ok := cl.Ping(ctx)
	require.True(t, ok)
	require.FileExists(t, logName)
}

func TestList(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)
	eOut := `+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|    BUS ID    |                  ID                  |  MODEL   | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0 | 5442d827-ff9a-5640-909e-f27c0455a451 | virt_gpu |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
| 0000:00:06.0 | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9 | virt_gpu |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`
	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestListDcgm(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockDcgmNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)
	eOut := `+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|    BUS ID    |                  ID                  |  MODEL   | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0 | 5442d827-ff9a-5640-909e-f27c0455a451 | virt_gpu |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |    55.1 |     38.1 |
| 0000:00:06.0 | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9 | virt_gpu |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |    99.2 |       96 |
+--------------+--------------------------------------+----------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`
	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestListOneMigDev(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockOneMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	eOut := `+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|     BUS ID      |                        ID                        |          MODEL          | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0.49 | MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0 | gpu_tesla_a100_mig_2c4g |   4096 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
| 0000:00:06.0    | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9             | virt_gpu                |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`

	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestListOneMigDevDcgm(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockDcgmOneMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	eOut := `+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|     BUS ID      |                        ID                        |          MODEL          | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0.49 | MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0 | gpu_tesla_a100_mig_2c4g |   4096 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |    10.1 |     22.3 |
| 0000:00:06.0    | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9             | virt_gpu                |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |    99.2 |       96 |
+-----------------+--------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`

	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestListTwoMigOnSameDev(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockTwoMigSameGpu)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	eOut := `+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|      BUS ID      |                        ID                         |          MODEL          | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0.121 | MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/13/0 | gpu_tesla_a100_mig_1c2g |   2048 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
| 0000:00:05.0.49  | MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0  | gpu_tesla_a100_mig_2c4g |   4096 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
| 0000:00:06.0     | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9              | virt_gpu                |  16384 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`

	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestListTwoMigOnDiffDev(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockTwoMigDiffGpu)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	eOut := `+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
|      BUS ID      |                        ID                         |          MODEL          | MEMORY | NUMA | DRIVER | CUDA | READY | THROTTLED | TEMP | USED | FREE | SM UTIL | SM OCCUP |
+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
| 0000:00:05.0.49  | MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0  | gpu_tesla_a100_mig_2c4g |   4096 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
| 0000:00:06.0.256 | MIG-GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4/13/0 | gpu_tesla_a100_mig_1c2g |   2048 |   -1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |      -1 |       -1 |
+------------------+---------------------------------------------------+-------------------------+--------+------+--------+------+-------+-----------+------+------+------+---------+----------+
`

	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, eOut, o)
	require.Equal(t, "", e)
}

func TestVfioList(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	err := modprobe.LoadModule("vfio-pci")
	require.NoError(t, err)

	//// Switch devices to vfio-pci mode
	api := device.PciLibQemuMock{}
	dev, err := api.NewPciDev("0000:00:05.0")
	require.NoError(t, err)
	require.NotNil(t, dev)

	ilog.Log().Debug("set driver for", zap.String("id", dev.UUID))
	grp, err := dev.SetDriver(ctx, "vfio-pci")
	require.NoError(t, err)
	require.NotNil(t, grp)

	// Actual test
	ep, lk := getDaemonConf(t.Name())
	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	eOut := `+--------------+--------------------------------------+----------+--------+------+-------------+--------+
|    BUS ID    |                  ID                  |  MODEL   | MEMORY | NUMA | IOMMU GROUP | ACTIVE |
+--------------+--------------------------------------+----------+--------+------+-------------+--------+
| 0000:00:05.0 | 5442d827-ff9a-5640-909e-f27c0455a451 | virt_gpu |  16384 |   -1 |           5 | false  |
+--------------+--------------------------------------+----------+--------+------+-------------+--------+
`
	o, e, err := run(ctx, clientPath, "list", "--show-vfio", "--service", ep)
	assert.NoError(t, err)
	assert.Equal(t, eOut, o)
	assert.Equal(t, "", e)

	// Restore origial driver binding
	grp, err = dev.SetDriver(ctx, "pci-stub")
	assert.NoError(t, err)
	assert.NotNil(t, grp)
}

func TestListJson(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)
	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--json", "--service", ep)
	require.NoError(t, err)
	require.NotEmpty(t, o)
	require.Equal(t, "", e)

	devices := []struct {
		Meta *pb.PciDeviceMeta `json:"meta"`
		Spec struct {
			PciDevice *pb.PciDeviceSpec `json:"pci_device"`
		} `json:"spec"`
		Status struct {
			Ready *pb.Condition `json:"ready"`
		} `json:"status"`
	}{}
	err = json.Unmarshal([]byte(o), &devices)
	require.NoError(t, err)
	fmt.Printf("devices:%v ", devices)
	require.NotEmpty(t, devices)
	require.Equal(t, len(devices), 2)
	for _, d := range devices {
		require.Equal(t, "virt_gpu", d.Spec.PciDevice.ModelName)
		require.Equal(t, "pci-stub", d.Spec.PciDevice.DriverName)
		require.Truef(t, d.Status.Ready.Status, "Bad device status for ID: "+d.Meta.Id)
	}
}

func TestAlloc(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	tests := []struct {
		name   string
		ids    []string
		driver string
		out    string
		eout   string
		fail   bool
	}{
		{
			name:   "single_dev",
			driver: "host",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			out:  "env='NVIDIA_VISIBLE_DEVICES=1' devices='/dev/vhost-vsock rw;/dev/fb1 rw'",
			fail: false,
		}, {
			name:   "two_dev",
			driver: "host",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
				"5442d827-ff9a-5640-909e-f27c0455a451",
			},
			out:  "env='NVIDIA_VISIBLE_DEVICES=1,0' devices='/dev/vhost-vsock rw;/dev/fb1 rw;/dev/fb0 rw'",
			fail: false,
		}, {
			name:   "bad_dev",
			driver: "host",
			ids: []string{
				"b2a4443c-d592-57ee-BAD-a4fc6dc7b837",
			},
			out:  "rpc error: code = NotFound desc = invalid allocation request: unknown device: b2a4443c-d592-57ee-BAD-a4fc6dc7b837\n",
			fail: true,
		}, {
			name:   "good_and_bad",
			driver: "host",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
				"b2a4443c-d592-57ee-BAD-a4fc6dc7b837",
			},
			out:  "rpc error: code = NotFound desc = invalid allocation request: unknown device: b2a4443c-d592-57ee-BAD-a4fc6dc7b837\n",
			fail: true,
		}, {
			name:   "good_dev_bad_mode",
			driver: "bad_mode",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			out:  "unsupporder driver type: bad_mode\n",
			fail: true,
		}, {
			name:   "good_dev_wrong_mode",
			driver: "vfio",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			out:  "rpc error: code = FailedPrecondition desc = invalid allocation request: wrong driver mode: 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9\n",
			fail: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			opts := []string{"--service", ep, "alloc", "--driver", tt.driver}
			opts = append(opts, tt.ids...)
			o, _, err := run(ctx, clientPath, opts...)
			if tt.fail {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.out, o)
			//require.Equal(t, tt.eout, e)
		})
	}

}

func TestAllocEnv(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	_, _, err := run(ctx, serverPath, "--setup-layers")
	require.NoError(t, err)

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	tests := []struct {
		name string
		ids  []string
		out  string
		eout string
		fail bool
	}{
		{
			name: "single_dev",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			out:  "env='CUDA_LIB_PATH=/libcuda-dir;NVIDIA_VISIBLE_DEVICES=1' bind='/opt/nvgpu-manager2/layers/libcuda-dir /libcuda-dir ro' devices='/dev/vhost-vsock rw;/dev/fb1 rw'",
			fail: false,
		},
		{
			name: "two_dev",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
				"5442d827-ff9a-5640-909e-f27c0455a451",
			},
			out:  "env='CUDA_LIB_PATH=/libcuda-dir;NVIDIA_VISIBLE_DEVICES=1,0' bind='/opt/nvgpu-manager2/layers/libcuda-dir /libcuda-dir ro' devices='/dev/vhost-vsock rw;/dev/fb1 rw;/dev/fb0 rw'",
			fail: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			opts := []string{"--service", ep, "alloc"}
			opts = append(opts, tt.ids...)
			o, _, err := run(ctx, clientPath, opts...)
			if tt.fail {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.out, o)
			//require.Equal(t, tt.eout, e)
		})
	}
	stopDaemon(daemon)
	_, _, err = run(ctx, serverPath, "--clean-layers")
	require.NoError(t, err)

	_, err = os.Stat("/opt/nvgpu-manager2/layers/libcuda")
	require.Error(t, err)
}

func TestAllocMig(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	_, _, err := run(ctx, serverPath, "--setup-layers")
	require.NoError(t, err)

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockOneMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	tests := []struct {
		name string
		ids  []string
		out  string
		eout string
		fail bool
	}{
		{
			name: "mig_dev",
			ids: []string{
				"MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0",
			},
			out:  "env='CUDA_LIB_PATH=/libcuda-dir;NVIDIA_VISIBLE_DEVICES=MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0' bind='/opt/nvgpu-manager2/layers/libcuda-dir /libcuda-dir ro' devices='/dev/vhost-vsock rw;/dev/fb0 rw;/dev/nvidia-caps/nvidia-cap48 rw;/dev/nvidia-caps/nvidia-cap49 rw'",
			fail: false,
		},
		{
			name: "parent_dev",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
			},
			out:  "rpc error: code = NotFound desc = invalid allocation request: unknown device: 5442d827-ff9a-5640-909e-f27c0455a451\n",
			fail: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			opts := []string{"--service", ep, "alloc"}
			opts = append(opts, tt.ids...)
			o, _, err := run(ctx, clientPath, opts...)
			if tt.fail {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
			require.Equal(t, tt.out, o)
			//require.Equal(t, tt.eout, e)
		})
	}
	stopDaemon(daemon)
	_, _, err = run(ctx, serverPath, "--clean-layers")
	require.NoError(t, err)

	_, err = os.Stat("/opt/nvgpu-manager2/layers/libcuda")
	require.Error(t, err)

}

func TestAllocVfio(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	err := modprobe.LoadModule("vfio-pci")
	require.NoError(t, err)

	//// Switch devices to vfio-pci mode
	api := device.PciLibQemuMock{}
	dev, err := api.NewPciDev("0000:00:05.0")
	require.NoError(t, err)
	require.NotNil(t, dev)

	ilog.Log().Debug("set driver for", zap.String("id", dev.UUID))
	grp, err := dev.SetDriver(ctx, "vfio-pci")
	require.NoError(t, err)
	require.NotNil(t, grp)

	// Actual test
	ep, lk := getDaemonConf(t.Name())
	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	tests := []struct {
		name   string
		driver string
		ids    []string
		out    string
		eout   string
		fail   bool
	}{
		{
			name: "single_dev",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
			},
			out:  "devices='/dev/vfio/vfio rw;/dev/vfio/5 rw'",
			fail: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			opts := []string{"--service", ep, "alloc", "--driver", "vfio"}
			opts = append(opts, tt.ids...)
			o, _, err := run(ctx, clientPath, opts...)
			if tt.fail {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}
			assert.Equal(t, tt.out, o)
		})
	}
	// Restore origial driver binding
	grp, err = dev.SetDriver(ctx, "pci-stub")
	assert.NoError(t, err)
	assert.NotNil(t, grp)
}

func TestPortoDevCheck(t *testing.T) {
	layerPath := yatest.BuildPath("infra/environments/rtc-xenial-gpu/release/vm-layer/layer.tar.zst")
	require.FileExists(t, layerPath)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	o, e, err := run(ctx, clientPath, "alloc", "9a86ef1e-93df-5bed-8bfb-cbb132cf88d9", "--service", ep)
	require.NoError(t, err)
	require.Equal(t, "env='NVIDIA_VISIBLE_DEVICES=1' devices='/dev/vhost-vsock rw;/dev/fb1 rw'", o)
	require.Equal(t, "", e)

	opts := stripTokens(o)
	// Porto layer import takes too long, disable it for ut case
	//_, _, err = run(ctx, "portoctl", "layer", "-I", "rtc-xenial-gpu", layerPath)
	//require.NoError(t, err)
	portoArgs := []string{
		"exec", "gpu-dev-check",
		//"layers=rtc-xenial-gpu",
		"command=python -c 'import os; print(os.access(\"/dev/fb1\", os.W_OK))'",
	}
	// W/o explicit device list /dev/fb1 should not be accessable
	o, e, err = run(ctx, "portoctl", portoArgs...)
	require.NoError(t, err)
	require.Equal(t, "False\n", o)
	require.Equal(t, "", e)

	// With devices option it should works
	portoArgs = append(portoArgs, opts...)
	o, e, err = run(ctx, "portoctl", portoArgs...)
	require.NoError(t, err)
	require.Equal(t, "True\n", o)
	require.Equal(t, "", e)

}

func TestYasmMetrics(t *testing.T) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	wantData := []string{
		"runtimecloud|gpu_driver=nvidia;gpu_model=virt_gpu;gpu_path=fb0",
		"runtimecloud|gpu_driver=nvidia;gpu_model=virt_gpu;gpu_path=fb1",
		"push-gpustat-device_count_tmmv",
	}

	for tries := 20; tries != 0; tries-- {
		time.Sleep(time.Second)
		resp, err := http.Get("http://localhost:11003/json")
		if err != nil {
			continue
		}
		if resp.StatusCode != 200 {
			continue
		}
		replyBody, _ := ioutil.ReadAll(resp.Body)
		_ = resp.Body.Close()
		data := string(replyBody)
		ilog.Log().Debug("got reply", zap.String("reply", data))
		found := false
		for _, t := range wantData {
			found = strings.Contains(data, t)
			ilog.Log().Info("lookup_token", zap.String("token", t), zap.Bool("found", found))
			if !found {
				break
			}

		}
		if !found {
			continue
		}
		// At this point we have found all patterns, success
		return

	}
	t.Fatal("No metrics found")
}

func TestSetDriver(t *testing.T) {
	ll := ilog.Log()

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep, lk := getDaemonConf(t.Name())

	daemon, err := startDaemon(ctx, ep, lk, "", config.MockNoMig)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	cl, err := client.NewClient(ll, ep)
	require.NoError(t, err)
	require.NotNil(t, cl)
	defer cl.Close()

	tests := []struct {
		name   string
		driver string
		ids    []string
		devMap map[string]string
		fail   bool
	}{
		//| 0000:00:05.0 | 5442d827-ff9a-5640-909e-f27c0455a451 | virt_gpu |  12206 |    0 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |
		//| 0000:00:06.0 | 9a86ef1e-93df-5bed-8bfb-cbb132cf88d9 | virt_gpu |  12206 |    1 | 418.67 | 10.1 | OK    | OK        |   40 | 1024 | 1024 |

		{
			name:   "dev2_to_vfio",
			driver: "vfio",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "vfio",
			},
			fail: false,
		}, {
			name:   "dev2_to_vfio_again",
			driver: "vfio",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "vfio",
			},
			fail: false,
		}, {
			name:   "dev2_to_host",
			driver: "host",
			ids: []string{
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "host",
			},
			fail: false,
		}, {
			name:   "all_to_vfio",
			driver: "vfio",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "vfio",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "vfio",
			},
			fail: false,
		}, {
			name:   "all_to_host",
			driver: "host",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "host",
			},
			fail: false,
		}, {
			name:   "bad_arg1",
			driver: "host",
			ids: []string{
				"5442d827-ff9a-5640-BAD-f27c0455a451",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "host",
			},
			fail: true,
		}, {
			name:   "bad_arg2",
			driver: "host",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
				"9a86ef1e-93df-5bed-BAD-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "host",
			},
			fail: true,
		}, {
			name:   "bad_driver",
			driver: "unknown_driver",
			ids: []string{
				"5442d827-ff9a-5640-909e-f27c0455a451",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			},
			devMap: map[string]string{
				"5442d827-ff9a-5640-909e-f27c0455a451": "host",
				"9a86ef1e-93df-5bed-8bfb-cbb132cf88d9": "host",
			},
			fail: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			devMap := make(map[string]string)
			opts := []string{"--service", ep, "set-driver", "--driver", tt.driver}

			opts = append(opts, tt.ids...)
			o, _, err := run(ctx, clientPath, opts...)
			if tt.fail {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
				require.Equal(t, "", o)
			}
			r, err := cl.Client.ListDevices(ctx, &pb.Empty{})
			require.NoError(t, err)
			ll.Debug("List", zap.Any("reply", r))

			for _, d := range r.Devices {
				require.Truef(t, d.Status.Ready.Status, "Bad device status for ID: "+d.Meta.Id)
				require.NotContainsf(t, devMap, d.Meta.Id, "Device Id collision for ID: "+d.Meta.Id)
				switch d.Spec.Driver.(type) {
				case *pb.GpuDeviceSpec_Nvidia:
					devMap[d.Meta.Id] = "host"
				case *pb.GpuDeviceSpec_Vfio:
					devMap[d.Meta.Id] = "vfio"
				}
			}
			ll.Debug("Compare dev list", zap.Any("want", tt.devMap), zap.Any("got", devMap))
			require.Equalf(t, tt.devMap, devMap, "Unexpected device configuration")
		})
	}
}
