package ut_test

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"os"
	"os/exec"
	"testing"
	"time"

	"a.yandex-team.ru/library/go/test/yatest"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	pb "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/client"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
)

var (
	serverPath string
	clientPath string
)

func init() {
	// TODO
	serverPath = yatest.BuildPath("infra/rsm/nvgpumanager/cmd/nvgpu-manager/nvgpu-manager")
	clientPath = yatest.BuildPath("infra/rsm/nvgpumanager/cmd/nvgpuctl/nvgpuctl")

	if _, err := os.Stat(serverPath); err != nil {
		panic("does not exists, path: " + serverPath)
	}
	if _, err := os.Stat(clientPath); err != nil {
		panic("does not exists path:" + clientPath)
	}
}

func startDaemon(ctx context.Context, ep string, lockPath, logFile string, isMock bool) (*exec.Cmd, error) {
	cmd := exec.CommandContext(ctx, serverPath,
		"--debug", "--service", ep, "--secure=false")
	if logFile != "" {
		cmd.Args = append(cmd.Args, "--log-file", logFile)
	}
	if isMock {
		cmd.Args = append(cmd.Args, "--mock-mode")
	}
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stdout

	err := cmd.Start()
	if err != nil {
		return nil, err
	}

	// Wait for server to start respond
	for tries := 10; tries != 0; tries-- {
		var cl *client.Client

		time.Sleep(time.Second * 1)
		cl, err = client.NewClient(ilog.Log(), ep)
		if err != nil {
			continue
		}
		if cl.Ping(ctx) {
			return cmd, nil
		}
		cl.Close()
	}
	return nil, err
}

func stopDaemon(c *exec.Cmd) {
	if c != nil {
		_ = c.Process.Kill()
		_ = c.Wait()
	}
}

func run(ctx context.Context, cmd string, arg ...string) (string, string, error) {
	c := exec.CommandContext(ctx, cmd, arg...)
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr
	err := c.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", c.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))
	return stdout.String(), stderr.String(), err

}

func assertYT(t *testing.T) {
	if os.Getenv("USE_YT_SPEC") != "1" {
		t.Skipf("'USE_YT_SPEC' is not defined, test should be run inside YT or distbuild")
	}
}

func TestListDev(t *testing.T) {
	assertYT(t)

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ep := "unix:gpuman-" + t.Name() + ".sock"
	lockPath := "gpuman-" + t.Name() + ".lock"
	logFile := "gpuman-" + t.Name() + ".log"

	daemon, err := startDaemon(ctx, ep, lockPath, logFile, false)
	defer stopDaemon(daemon)
	require.NoError(t, err)

	o, e, err := run(ctx, clientPath, "list", "--show-nvgpu", "--json", "--service", ep)
	require.NoError(t, err)
	require.NotEmpty(t, o)
	require.Equal(t, "", e)
	require.FileExists(t, logFile)
	devices := []struct {
		Meta *pb.PciDeviceMeta `json:"meta"`
		Spec struct {
			PciDevice *pb.PciDeviceSpec `json:"pci_device"`
		} `json:"spec"`
		Status struct {
			Ready *pb.Condition `json:"ready"`
		} `json:"status"`
	}{}
	err = json.Unmarshal([]byte(o), &devices)
	require.NoError(t, err)
	fmt.Printf("devices:%v ", devices)
	require.NotEmpty(t, devices)
	require.Equal(t, len(devices), 1)
	for _, d := range devices {
		require.Equal(t, "gpu_geforce_1080ti", d.Spec.PciDevice.ModelName)
		require.Equal(t, "nvidia", d.Spec.PciDevice.DriverName)
		require.Truef(t, d.Status.Ready.Status, "Bad device status for ID: "+d.Meta.Id)
	}
}
