package cmd

import (
	"context"
	"fmt"
	"os"

	"github.com/spf13/cobra"
	"go.uber.org/zap"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/client"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/config"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
	"a.yandex-team.ru/library/go/core/buildinfo"
)

var (
	cfgFile         string
	debug           bool
	serviceEndpoint string
	tracerEndpoint  string
	tracerEnable    bool
	dumpJSON        bool
	apiClient       *client.Client
	apiCtx          context.Context
)

var rootCmd = &cobra.Command{
	Use:   "nvgpuctl",
	Short: "NVIDIA GPU Manager client",
	Long:  "NVIDIA GPU manager client, Docs https://doc.yandex-team.ru/nvgpu_manager",
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		return initClient()
	},
	PersistentPostRunE: func(cmd *cobra.Command, args []string) error {
		return destroyClient()
	},
	Version: buildinfo.Info.ProgramVersion,
}

// Execute adds all child commands to the root command and sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Println(err)
		os.Exit(1)
	}
}

func init() {
	if buildinfo.Info.ProgramVersion == "" {
		rootCmd.Version = "N/A"
	} else {
		rootCmd.Version = buildinfo.Info.ProgramVersion
	}

	rootCmd.PersistentFlags().BoolVarP(&debug, "debug", "d", false, "enable debug level")
	rootCmd.PersistentFlags().StringVarP(&serviceEndpoint, "service", "S", config.DefaultServerAddress, "service GRPC endpoint")

	rootCmd.PersistentFlags().BoolVar(&tracerEnable, "enable-opentrace", false, "enable opentrace")
	rootCmd.PersistentFlags().StringVar(&tracerEndpoint, "opentrace-endpoint", "", "opentrace endpoint url")
}

func initClient() error {
	var err error
	ilog.Init()
	ll := ilog.Log()
	defer ll.Sync()
	ilog.Cfg().Level.SetLevel(zap.WarnLevel)
	if debug {
		ilog.Cfg().Level.SetLevel(zap.DebugLevel)
	}
	apiClient, err = client.NewClient(ll, serviceEndpoint)
	if err != nil {
		return err
	}
	md := metadata.Pairs("x-client-id", fmt.Sprintf("nvgpuctl:%d", os.Getpid()))
	apiCtx = metadata.NewOutgoingContext(context.Background(), md)
	return nil
}

func destroyClient() error {
	ilog.Log().Debug("destroy client", zap.Any("client", apiClient))
	apiClient.Close()
	ilog.Log().Debug("destroy client done")
	// If log is os.Stdout it has no sync method, just ignore error
	_ = ilog.Log().Sync()
	return nil
}
