package device

import (
	"bufio"
	"os"
	"strconv"
	"strings"
)

const (
	NvidiaMinorsConfigPath = "/proc/driver/nvidia-caps/mig-minors"
)

type GpuInstanceCaps struct {
	OwnCap               uint32
	ComputeInstancesCaps map[uint32]uint32
}

type NvidiaCapsConfiguration struct {
	instancesMap map[uint32]map[uint32]GpuInstanceCaps
}

func NewNvidiaCapsConfiguration() (*NvidiaCapsConfiguration, error) {

	capsConfig := &NvidiaCapsConfiguration{instancesMap: make(map[uint32]map[uint32]GpuInstanceCaps)}

	// check file exists

	// read file
	file, err := os.Open(NvidiaMinorsConfigPath)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	scanner := bufio.NewScanner(file)

	for scanner.Scan() {
		baseSplited := strings.Split(scanner.Text(), " ")
		pathSplited := strings.Split(baseSplited[0], "/")
		switch len(pathSplited) {
		case 3:

			var gpuID64, giID64, minor64 uint64
			var err error

			gpuID64, err = strconv.ParseUint(pathSplited[0][3:], 10, 32)
			if err != nil {
				return nil, err
			}

			giID64, err = strconv.ParseUint(pathSplited[1][2:], 10, 32)
			if err != nil {
				return nil, err
			}

			minor64, err = strconv.ParseUint(baseSplited[1], 10, 32)
			if err != nil {
				return nil, err
			}

			gpuID := uint32(gpuID64)
			giID := uint32(giID64)
			minor := uint32(minor64)

			if _, ok := capsConfig.instancesMap[gpuID]; !ok {
				capsConfig.instancesMap[gpuID] = make(map[uint32]GpuInstanceCaps)
			}
			capsConfig.instancesMap[gpuID][giID] = GpuInstanceCaps{OwnCap: minor, ComputeInstancesCaps: make(map[uint32]uint32)}
		case 4:

			var gpuID64, giID64, ciID64, minor64 uint64
			var err error

			gpuID64, err = strconv.ParseUint(pathSplited[0][3:], 10, 32)
			if err != nil {
				return nil, err
			}

			giID64, err = strconv.ParseUint(pathSplited[1][2:], 10, 32)
			if err != nil {
				return nil, err
			}

			ciID64, err = strconv.ParseUint(pathSplited[2][2:], 10, 32)
			if err != nil {
				return nil, err
			}

			minor64, err = strconv.ParseUint(baseSplited[1], 10, 32)
			if err != nil {
				return nil, err
			}

			gpuID := uint32(gpuID64)
			giID := uint32(giID64)
			ciID := uint32(ciID64)
			minor := uint32(minor64)

			capsConfig.instancesMap[gpuID][giID].ComputeInstancesCaps[ciID] = minor
		}
	}

	if err = scanner.Err(); err != nil {
		return nil, err
	}

	return capsConfig, nil
}

func (capsConf NvidiaCapsConfiguration) GetGpuInstanceMinor(deviceMinor uint32, GpuInstanceID uint32) uint32 {
	instanceCapsMap, ok := capsConf.instancesMap[deviceMinor]
	if !ok {
		return 0
	}

	var instanceCaps GpuInstanceCaps
	instanceCaps, ok = instanceCapsMap[GpuInstanceID]
	if !ok {
		return 0
	}

	return instanceCaps.OwnCap
}

func (capsConf NvidiaCapsConfiguration) GetComputeInstanceMinor(deviceMinor uint32, GpuInstanceID uint32, computeInstanceID uint32) uint32 {
	instanceCapsMap, ok := capsConf.instancesMap[deviceMinor]
	if !ok {
		return 0
	}

	var gInstanceCaps GpuInstanceCaps
	gInstanceCaps, ok = instanceCapsMap[GpuInstanceID]
	if !ok {
		return 0
	}

	var cInstanceCap uint32
	cInstanceCap, ok = gInstanceCaps.ComputeInstancesCaps[computeInstanceID]
	if !ok {
		return 0
	}

	return cInstanceCap
}
