package device

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"testing"

	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/config"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
)

func TestMockNvmlDevice(t *testing.T) {
	tdir, err := ioutil.TempDir("/tmp/", "MockSysfs")
	require.NoError(t, err)
	defer os.RemoveAll(tdir)

	doMakeMockSysfs(t, tdir)
	pciMock := PciLibMockSysfs{
		BasePath: path.Join(tdir, "sys/bus/pci/devices"),
		Vendor:   "10de",
	}
	nvmlMock := NewNvmlMock(&pciMock, func(p *PciDevice) bool {
		return p.Vendor == "10de"
	}, "no-mig")

	pciDevices, err := pciMock.NewPciDevices()
	require.NoError(t, err)
	require.NotEmpty(t, pciDevices)
	require.Equal(t, mockPciDevices, pciDevices)

	pmap := make(map[string]*PciDevice)
	for _, d := range pciDevices {
		pmap[d.BusID] = d
	}

	var c config.Configuration
	config.ApplyOptions(&c, config.ApplyDefaultConfig())

	nvDevices, badDevsCnt, err := NewNvmlDevices(nvmlMock, nil, pmap, &c)
	require.Zero(t, badDevsCnt)
	require.NoError(t, err)
	ilog.Log().Debug("newNvmlDevices", zap.Any("devices", nvDevices))

	require.Equal(t, 4, len(nvDevices))

	for idx, d := range nvDevices {
		t.Run(fmt.Sprintf("nvidia%d", idx), func(t *testing.T) {
			ndev, err := nvmlMock.NewDevice(int(idx))
			require.NoError(t, err)
			status, err := nvmlMock.DeviceStatus(ndev)
			require.NoError(t, err)

			require.Equal(t, uint(idx), d.Index)
			require.Equal(t, mockPciDevices[idx], d.PciDev)
			require.Equal(t, ndev, d.Device)
			require.Equal(t, status, d.Status)
		})
	}
}
