package device

import (
	"fmt"
	"os"
	"sort"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/vendor/github.com/NVIDIA/go-nvml/pkg/nvml"
	// "github.com/gofrs/uuid"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/config"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/utils"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/modprobe"
)

type PciFilterFn func(*PciDevice) bool

func generateDevices(mockMode string) ([]FullNvmlAPIDevice, error) {
	k40 := string("Fake Tesla model")
	k40Power := uint32(235)
	k40Memory := uint64(16 << 10)

	baseObject := FullNvmlAPIDevice{
		Model:   k40,
		Power:   k40Power,
		Memory:  k40Memory,
		MigMode: false,
	}

	switch mockMode {
	case config.MockNoMig, config.MockDcgmNoMig:

		return utils.GenerateObjects(
			baseObject,
			[]map[string]interface{}{
				{"UUID": "GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae", "CPUAffinity": 0, "Path": "/dev/fb0"},
				{"UUID": "GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4", "CPUAffinity": 1, "Path": "/dev/fb1"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 0, "Path": "/dev/fb2"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 1, "Path": "/dev/fb3"},
				{"UUID": "GPU-d5484dd2-fe83-41f3-a77f-80781513ec69", "CPUAffinity": 0, "Path": "/dev/fb4"},
				{"UUID": "GPU-77ad526b-868f-412a-843c-cfdc8711dc9f", "CPUAffinity": 1, "Path": "/dev/fb5"},
				{"UUID": "GPU-c9b04426-926e-4506-b500-dea4c3bb5919", "CPUAffinity": 0, "Path": "/dev/fb6"},
				{"UUID": "GPU-4c9197a1-ff63-4275-ad6e-42526c9654e3", "CPUAffinity": 1, "Path": "/dev/fb7"},
			})

	case config.MockOneMig, config.MockDcgmOneMig:

		ci := &NvComputeInstance{InstanceMinor: 49, Path: "/dev/nvidia-caps/nvidia-cap49"}
		gi := &NvGpuInstance{Path: "/dev/nvidia-caps/nvidia-cap48", ComputeInstances: map[uint32]*NvComputeInstance{0: ci}}

		result, err := utils.GenerateObjects(
			baseObject,
			[]map[string]interface{}{
				{
					"UUID":         "GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae",
					"CPUAffinity":  0,
					"Path":         "/dev/fb0",
					"MigMode":      true,
					"GpuInstances": map[uint32]*NvGpuInstance{5: gi},
					"MigDevices": []*MigNvmlAPIDevice{&MigNvmlAPIDevice{
						GpuInstance:     gi,
						ComputeInstance: ci,
						UUID:            "MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0",
						Model:           "A100-SXM-80GB MIG 2g.4gb",
						Memory:          4096,
					}}},
				{"UUID": "GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4", "CPUAffinity": 1, "Path": "/dev/fb1"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 0, "Path": "/dev/fb2"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 1, "Path": "/dev/fb3"},
				{"UUID": "GPU-d5484dd2-fe83-41f3-a77f-80781513ec69", "CPUAffinity": 0, "Path": "/dev/fb4"},
				{"UUID": "GPU-77ad526b-868f-412a-843c-cfdc8711dc9f", "CPUAffinity": 1, "Path": "/dev/fb5"},
				{"UUID": "GPU-c9b04426-926e-4506-b500-dea4c3bb5919", "CPUAffinity": 0, "Path": "/dev/fb6"},
				{"UUID": "GPU-4c9197a1-ff63-4275-ad6e-42526c9654e3", "CPUAffinity": 1, "Path": "/dev/fb7"},
			})

		if err != nil {
			return []FullNvmlAPIDevice{}, err
		}

		result[0].MigDevices[0].ParentDevice = &result[0]

		return result, err

	case config.MockTwoMigSameGpu:

		ci1 := &NvComputeInstance{InstanceMinor: 49, Path: "/dev/nvidia-caps/nvidia-cap49"}
		gi1 := &NvGpuInstance{Path: "/dev/nvidia-caps/nvidia-cap48", ComputeInstances: map[uint32]*NvComputeInstance{0: ci1}}

		ci2 := &NvComputeInstance{InstanceMinor: 121, Path: "/dev/nvidia-caps/nvidia-cap121"}
		gi2 := &NvGpuInstance{Path: "/dev/nvidia-caps/nvidia-cap120", ComputeInstances: map[uint32]*NvComputeInstance{0: ci2}}

		result, err := utils.GenerateObjects(
			baseObject,
			[]map[string]interface{}{
				{
					"UUID":         "GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae",
					"CPUAffinity":  0,
					"Path":         "/dev/fb0",
					"MigMode":      true,
					"GpuInstances": map[uint32]*NvGpuInstance{5: gi1, 13: gi2},
					"MigDevices": []*MigNvmlAPIDevice{
						&MigNvmlAPIDevice{
							GpuInstance:     gi1,
							ComputeInstance: ci1,
							UUID:            "MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0",
							Model:           "A100-SXM-80GB MIG 2g.4gb",
							Memory:          4096,
						},
						&MigNvmlAPIDevice{
							GpuInstance:     gi2,
							ComputeInstance: ci2,
							UUID:            "MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/13/0",
							Model:           "A100-SXM-80GB MIG 1g.2gb",
							Memory:          2048,
						},
					}},
				{"UUID": "GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4", "CPUAffinity": 1, "Path": "/dev/fb1"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 0, "Path": "/dev/fb2"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 1, "Path": "/dev/fb3"},
				{"UUID": "GPU-d5484dd2-fe83-41f3-a77f-80781513ec69", "CPUAffinity": 0, "Path": "/dev/fb4"},
				{"UUID": "GPU-77ad526b-868f-412a-843c-cfdc8711dc9f", "CPUAffinity": 1, "Path": "/dev/fb5"},
				{"UUID": "GPU-c9b04426-926e-4506-b500-dea4c3bb5919", "CPUAffinity": 0, "Path": "/dev/fb6"},
				{"UUID": "GPU-4c9197a1-ff63-4275-ad6e-42526c9654e3", "CPUAffinity": 1, "Path": "/dev/fb7"},
			})

		if err != nil {
			return []FullNvmlAPIDevice{}, err
		}

		result[0].MigDevices[0].ParentDevice = &result[0]
		result[0].MigDevices[1].ParentDevice = &result[0]

		return result, err

	case config.MockTwoMigDiffGpu:

		ci1 := &NvComputeInstance{InstanceMinor: 49, Path: "/dev/nvidia-caps/nvidia-cap49"}
		gi1 := &NvGpuInstance{Path: "/dev/nvidia-caps/nvidia-cap48", ComputeInstances: map[uint32]*NvComputeInstance{0: ci1}}

		ci2 := &NvComputeInstance{InstanceMinor: 256, Path: "/dev/nvidia-caps/nvidia-cap256"}
		gi2 := &NvGpuInstance{Path: "/dev/nvidia-caps/nvidia-cap255", ComputeInstances: map[uint32]*NvComputeInstance{0: ci2}}

		result, err := utils.GenerateObjects(
			baseObject,
			[]map[string]interface{}{
				{
					"UUID":         "GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae",
					"CPUAffinity":  0,
					"Path":         "/dev/fb0",
					"MigMode":      true,
					"GpuInstances": map[uint32]*NvGpuInstance{5: gi1},
					"MigDevices": []*MigNvmlAPIDevice{
						&MigNvmlAPIDevice{
							GpuInstance:     gi1,
							ComputeInstance: ci1,
							UUID:            "MIG-GPU-b6c54489-38a0-5f50-a60a-fd8d76219cae/5/0",
							Model:           "A100-SXM-80GB MIG 2g.4gb",
							Memory:          4096,
						},
					}},
				{
					"UUID":         "GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4",
					"CPUAffinity":  1,
					"Path":         "/dev/fb1",
					"MigMode":      true,
					"GpuInstances": map[uint32]*NvGpuInstance{13: gi2},
					"MigDevices": []*MigNvmlAPIDevice{
						&MigNvmlAPIDevice{
							GpuInstance:     gi2,
							ComputeInstance: ci2,
							UUID:            "MIG-GPU-11116e73-1c03-5de6-9130-5f9925ae8ab4/13/0",
							Model:           "A100-SXM-80GB MIG 1g.2gb",
							Memory:          2048,
						},
					}},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 0, "Path": "/dev/fb2"},
				{"UUID": "GPU-5c15464f-1603-4efa-9056-269193e83f76", "CPUAffinity": 1, "Path": "/dev/fb3"},
				{"UUID": "GPU-d5484dd2-fe83-41f3-a77f-80781513ec69", "CPUAffinity": 0, "Path": "/dev/fb4"},
				{"UUID": "GPU-77ad526b-868f-412a-843c-cfdc8711dc9f", "CPUAffinity": 1, "Path": "/dev/fb5"},
				{"UUID": "GPU-c9b04426-926e-4506-b500-dea4c3bb5919", "CPUAffinity": 0, "Path": "/dev/fb6"},
				{"UUID": "GPU-4c9197a1-ff63-4275-ad6e-42526c9654e3", "CPUAffinity": 1, "Path": "/dev/fb7"},
			})

		if err != nil {
			return []FullNvmlAPIDevice{}, err
		}

		result[0].MigDevices[0].ParentDevice = &result[0]
		result[1].MigDevices[0].ParentDevice = &result[1]

		return result, err

	default:
		return []FullNvmlAPIDevice{}, nil
	}
}

// NvmlMock : Implementation of NvmlInterface using mocked calls
type NvmlMock struct {
	pciProvider PciInterface
	check       PciFilterFn
	devices     []FullNvmlAPIDevice
}

func NewNvmlMock(p PciInterface, filter PciFilterFn, mockMode string) *NvmlMock {

	devs, err := generateDevices(mockMode)
	fmt.Println(mockMode, err)

	return &NvmlMock{
		pciProvider: p,
		check:       filter,
		devices:     devs,
	}
}

func NewNvmlQemuMock(p PciInterface, mockMode string) *NvmlMock {

	devs, err := generateDevices(mockMode)
	fmt.Println(mockMode, err)

	return &NvmlMock{
		pciProvider: p,
		check:       func(d *PciDevice) bool { return d.Vendor == "1234" && d.Device == "1111" && d.Driver == "pci-stub" },
		devices:     devs,
	}
}

func (nm *NvmlMock) Init(c *config.Configuration) error {
	return nil
}

func (nm *NvmlMock) Shutdown() error {
	return nil
}

func (nm *NvmlMock) GetDeviceCount() (int, error) {
	nr := int(0)
	devices, err := nm.pciProvider.NewPciDevices()
	if err != nil {
		return 0, err
	}
	// Device should not be visiable if it is not controlled by nvidia driver,
	// for example it is used by fvio-pci, or model-id is not good
	for _, d := range devices {
		if !nm.check(d) {
			ilog.Log().Debug("Skip non nvidia device", zap.Any("pcidev", d))
			continue
		}
		nr++
	}
	return nr, nil
}

func (nm *NvmlMock) NewDevice(id int) (*FullNvmlAPIDevice, error) {
	var pdev *PciDevice
	nr := int(0)

	devices, err := nm.pciProvider.NewPciDevices()
	if err != nil {
		return nil, fmt.Errorf("bad device idx:%d, err:%w ", id, err)
	}
	// sort device list in order to make device-idx stable
	sort.Slice(devices[:], func(i, j int) bool {
		return devices[i].BusID < devices[j].BusID
	})

	for _, d := range devices {
		if !nm.check(d) {
			ilog.Log().Debug("Skip non nvidia device", zap.Any("pcidev", d))
			continue
		}
		if nr == id {
			pdev = d
			break
		}
		nr++
	}
	if pdev == nil {
		return nil, fmt.Errorf("bad device idx:%d, no such pcidevice %w", id, ErrNoent)
	}

	dev := &nm.devices[id]
	dev.PCIBusID = "0000" + pdev.BusID

	return dev, nil
}

func (nm *NvmlMock) DeviceStatus(device NvmlAPIDevice) (*NvmlAPIDeviceStatus, error) {
	power := uint32(250)
	temp := uint32(40)
	one := uint32(1)
	kilo := uint64(1024)
	proc := nvml.ProcessInfo{
		Pid: 123,
		// Name:       "gpu-user",
		UsedGpuMemory: 10240,
		// Type:       nvml.Compute,
	}
	status := NvmlAPIDeviceStatus{
		Power:       power,
		Temperature: temp,
		Clocks: ClockInfo{
			Cores:  one,
			Memory: one,
		},
		Utilization: UtilizationInfo{
			GPU:     one,
			Memory:  one,
			Encoder: one,
			Decoder: one,
		},
		Memory: MemoryInfo{
			Global: nvml.Memory{
				Used: kilo,
				Free: kilo,
			},
			ECCErrors: nvml.EccErrorCounts{
				L1Cache:      kilo,
				L2Cache:      kilo,
				DeviceMemory: kilo,
			},
		},
		Processes: []nvml.ProcessInfo{proc},
		Throttle:  nvml.ClocksThrottleReasonGpuIdle,
		PCI: PCIStatusInfo{
			BAR1Used: kilo,
			Throughput: PCIThroughputInfo{
				RX: one,
				TX: one,
			},
		},
	}
	return &status, nil
}

// GetDriverName : Return the driver name
func (nm *NvmlMock) GetDriverName() string {
	return "pci-stub"
}

func (nm *NvmlMock) GetDriverVersion() (string, error) {
	return "418.67", nil
}

func (nm *NvmlMock) GetCudaDriverVersion() (uint, uint, error) {
	major := uint(10)
	minor := uint(1)
	return major, minor, nil
}

// GetCtlDevices returns devices required for GPU management
func (nm *NvmlMock) GetCtlDevices(c *config.Configuration) ([]string, error) {
	// Pick vhost-sock as random device just for
	return []string{"/dev/vhost-vsock"}, nil
}

// EnableDriver load pci-stub kernel driver
func (nm *NvmlMock) EnableDriver(services []*utils.Service) error {
	var err error
	// Check if we module was blacklisted
	if _, err = os.Stat(NvidiaBlackListConfig); err == nil {
		err = os.Remove(NvidiaBlackListConfig)
		if err != nil {
			return err
		}
	}
	return modprobe.LoadModule(nm.GetDriverName())
}

// doDisableDriver blacklist and unload nvidia kernel driver
func (nm *NvmlMock) DisableDriver(services []*utils.Service) error {
	var err error
	m := nm.GetDriverName()

	err = modprobe.BlacklistModule(m, NvidiaBlackListConfig)
	if err != nil {
		_ = os.Remove(NvidiaBlackListConfig)
		return err
	}
	err = modprobe.UnloadModule(m)
	if err != nil {
		_ = os.Remove(NvidiaBlackListConfig)
		return err
	}
	return nil
}
