package device

import (
	"fmt"
	"path"
)

type PciLibMockSysfs struct {
	BasePath string
	Vendor   string
}

type PciLibNvidiaMock struct {
	DevMap map[string]*PciDevice
}

func (pl *PciLibMockSysfs) NewPciDev(busID string) (*PciDevice, error) {
	return newPciDev(path.Join(pl.BasePath, busID))
}
func (pl *PciLibMockSysfs) NewPciDevices() ([]*PciDevice, error) {
	return newPciDevices(pl.BasePath, pl.Vendor, PciGpuClass)
}

func NewPciNvidiaMock() *PciLibNvidiaMock {
	m := make(map[string]*PciDevice)

	dl := []*PciDevice{
		&PciDevice{
			BusID:        "0000:03:00.0",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "GK110BGL [Tesla K40m]",
			Driver:       "nvidia",
			NumaNode:     0,
		},
		&PciDevice{
			BusID:        "0000:04:00.0",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "GK110BGL [Tesla K40m]",
			Driver:       "nvidia",
			NumaNode:     0,
		},
		&PciDevice{
			BusID:        "0000:83:00.0",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "GK110BGL [Tesla K40m]",
			Driver:       "nvidia",
			NumaNode:     1,
		},
		&PciDevice{
			BusID:        "0000:84:00.0",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "GK110BGL [Tesla K40m]",
			Driver:       "nvidia",
			NumaNode:     1,
		},
	}
	for _, d := range dl {
		m[d.BusID] = d
	}
	return &PciLibNvidiaMock{DevMap: m}
}

func (pl *PciLibNvidiaMock) NewPciDev(busID string) (*PciDevice, error) {
	d := new(PciDevice)
	tmpl, ok := pl.DevMap[busID]
	if !ok {
		return nil, fmt.Errorf("bad busid:%s %w", busID, ErrNoent)
	}

	*d = *tmpl
	d.GenUUID()
	return d, nil
}

func (pl *PciLibNvidiaMock) NewPciDevices() ([]*PciDevice, error) {
	devices := []*PciDevice{}

	for bus := range pl.DevMap {
		d, err := pl.NewPciDev(bus)
		if err != nil {
			return nil, err
		}
		devices = append(devices, d)
	}
	return devices, nil
}
