package device

import (
	"io/ioutil"
	"os"
	"os/exec"
	"path"
	"testing"

	"github.com/stretchr/testify/require"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
	"a.yandex-team.ru/library/go/core/resource"
)

var (
	mockPciDevices = []*PciDevice{
		&PciDevice{
			UUID:         "8fa5bc1f-9eba-502e-a31c-e5e38a1302c5",
			BusID:        "0000:03:00.0",
			Class:        "030200",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "gpu_tesla_k40",
			MemoryGb:     12,
			Driver:       "",
			NumaNode:     0,
			CurLinkSpeed: 2.5,
		},
		&PciDevice{
			UUID:         "95c5a0f7-7995-54dc-9eea-015135634d3e",
			BusID:        "0000:04:00.0",
			Class:        "030200",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "gpu_tesla_k40",
			MemoryGb:     12,
			Driver:       "",
			NumaNode:     0,
			CurLinkSpeed: 5,
		},
		&PciDevice{
			UUID:         "a7a863fd-1365-57e2-b319-8aab67001e0c",
			BusID:        "0000:83:00.0",
			Class:        "030200",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "gpu_tesla_k40",
			MemoryGb:     12,
			Driver:       "",
			NumaNode:     1,
			CurLinkSpeed: 8,
		},
		&PciDevice{
			UUID:         "a01d59ed-d918-573e-a299-6b2aa49295aa",
			BusID:        "0000:84:00.0",
			Class:        "030200",
			Vendor:       "10de",
			Device:       "1023",
			SubSysVendor: "10de",
			SubSysDevice: "097e",
			ModelName:    "gpu_tesla_k40",
			MemoryGb:     12,
			Driver:       "",
			NumaNode:     1,
			CurLinkSpeed: 8,
		},
	}
)

func init() {
	MockHWNamespace()

}

func TestGetNvidiaDevices(t *testing.T) {
	api := PciLibNvidia{}
	devices, err := api.NewPciDevices()
	require.NoError(t, err)
	require.Emptyf(t, devices, "Test excepts to run inside environments w/o nvidia devices, got:%v", devices)
}

func TestGetQemuGpuDevices(t *testing.T) {
	ll := ilog.Log()

	api := PciLibQemuMock{}
	devices, err := api.NewPciDevices()
	ll.Debug("dump devices", zap.Any("devices", devices))

	wantDev := []*PciDevice{
		&PciDevice{
			UUID:         "5442d827-ff9a-5640-909e-f27c0455a451",
			BusID:        "0000:00:05.0",
			Class:        "038000",
			Vendor:       "1234",
			Device:       "1111",
			SubSysVendor: "1af4",
			SubSysDevice: "1100",
			ModelName:    "virt_gpu",
			MemoryGb:     16,
			Driver:       "pci-stub",
			NumaNode:     -1,
		},
		&PciDevice{
			UUID:         "9a86ef1e-93df-5bed-8bfb-cbb132cf88d9",
			BusID:        "0000:00:06.0",
			Class:        "038000",
			Vendor:       "1234",
			Device:       "1111",
			SubSysVendor: "1af4",
			SubSysDevice: "1100",
			ModelName:    "virt_gpu",
			MemoryGb:     16,
			Driver:       "pci-stub",
			NumaNode:     -1,
		},
	}
	require.NoError(t, err)
	require.NotEmpty(t, devices)
	require.Equal(t, wantDev, devices)
}

func TestParseAllPci(t *testing.T) {
	ll := ilog.Log()

	devices, err := newPciDevices("/sys/bus/pci/devices", "", nil)
	require.NoError(t, err)
	require.NotEmpty(t, devices)

	for _, d := range devices {
		ll.Debug("iter", zap.Any("pcidev", d))
	}
}

func doMakeMockSysfs(t *testing.T, dir string) {

	data := resource.MustGet("/sas2-8192-pci-devices.tar.gz")
	err := ioutil.WriteFile("archive.tar.gz", data, 0644)
	require.NoError(t, err)

	c := exec.Command("tar", "-zxf", "archive.tar.gz", "-C", dir)
	c.Stdout = os.Stdout
	c.Stderr = os.Stderr
	err = c.Run()
	require.NoError(t, err)
}

func TestMockPciDevices(t *testing.T) {
	tdir, err := ioutil.TempDir("/tmp", "MockSysfs")
	require.NoError(t, err)
	defer os.RemoveAll(tdir)

	doMakeMockSysfs(t, tdir)
	pciMock := PciLibMockSysfs{
		BasePath: path.Join(tdir, "sys/bus/pci/devices"),
		Vendor:   "10de",
	}

	devices, err := pciMock.NewPciDevices()
	require.NoError(t, err)
	require.NotEmpty(t, devices)
	require.Equal(t, mockPciDevices, devices)
}
