package device

import (
	"fmt"
	"os"

	pb "a.yandex-team.ru/infra/rsm/nvgpumanager/api"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/utils"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/yasm"
)

type VFioPciDevice struct {
	Ready  pb.Condition
	PciDev *PciDevice
	Group  *IOMMUGroup
	Active bool
}

func (d *VFioPciDevice) ProtoMarshal() *pb.GpuDevice {
	return &pb.GpuDevice{
		Meta: &pb.PciDeviceMeta{
			Id:    d.PciDev.UUID,
			BusId: d.PciDev.BusID,
		},
		Spec: &pb.GpuDeviceSpec{
			PciDevice: d.PciDev.ProtoMarshal(),
			Driver: &pb.GpuDeviceSpec_Vfio{
				Vfio: &pb.VFioGpuSpec{
					IommuGroup: uint32(d.Group.ID),
				},
			},
		},
		Status: &pb.GpuDeviceStatus{
			Ready: &d.Ready,
			Driver: &pb.GpuDeviceStatus_Vfio{
				Vfio: &pb.VFioGpuStatus{
					Active: d.Active,
				},
			},
		},
	}
}
func (d *VFioPciDevice) YasmMetrics() yasm.YasmMetrics {
	gpuPath := d.Group.String()

	m := yasm.YasmMetrics{
		Tags: map[string]string{
			"itype":      "runtimecloud",
			"gpu_model":  d.PciDev.ModelName,
			"gpu_path":   gpuPath,
			"gpu_driver": "vfio-pci",
		},
		TTL: 30,
		Values: []yasm.YasmValue{
			yasm.YasmValue{
				Name:  "gpustat-device_ready_tmmv",
				Value: d.Ready.Status,
			},
			yasm.YasmValue{
				Name:  "gpustat-device_active_tmmv",
				Value: d.Active,
			},
		},
	}
	return m
}

// TransferStatus status from old object generation
func (d *VFioPciDevice) TransferStatus(orig *VFioPciDevice) {
	// If new state is Ready, we should inherent old state
	if d.Ready.Status {
		utils.CopyCondition(&d.Ready, &orig.Ready)
	}
}

func isGpuActive(linkSpeed float32) bool {
	return linkSpeed > 2.5
}

// NewVFioPciDevices returns list of vfio-pci devices
func NewVFioPciDevices(pcache map[string]*PciDevice) ([]*VFioPciDevice, error) {
	var devices []*VFioPciDevice

	for _, d := range pcache {
		if d.Driver != "vfio-pci" {
			continue
		}

		grp, err := NewIOMMUGroup(d.BusID)
		vdev := &VFioPciDevice{
			PciDev: d,
			Group:  grp,
		}
		if err != nil {
			utils.SetCondition(&vdev.Ready, false, "IOMMU not found")
		} else {
			utils.SetCondition(&vdev.Ready, true, "")
		}
		if vdev.PciDev.MemoryGb == 0 {
			utils.SetCondition(&vdev.Ready, false, "Unknown device")
		}

		vdev.Active = isGpuActive(vdev.PciDev.CurLinkSpeed)

		devices = append(devices, vdev)
	}
	return devices, nil
}

var pciNvswitchClass = map[string]string{
	"068000": "Bridge",
}

func GetVfioNvswitchDevices() ([]string, error) {
	return getVfioDevices("10de", pciNvswitchClass)
}

var pciMlxIbClass = map[string]string{
	"020700": "Infiniband controller",
}

func GetVfioMlxIbDevices() ([]string, error) {
	return getVfioDevices("15b3", pciMlxIbClass)
}

func getVfioDevices(vendor string, class map[string]string) ([]string, error) {
	devs := []string{}

	pciDevs, err := newPciDevices("/sys/bus/pci/devices", vendor, class)
	if err != nil {
		return devs, fmt.Errorf("failed to get vfio pci devices with vendor_id=%s", vendor)
	}

	for _, pciDev := range pciDevs {
		g, err := NewIOMMUGroup(pciDev.BusID)
		if err != nil {
			return devs, fmt.Errorf("failed to get iommu group for %s device", pciDev.BusID)
		}

		vfioDevPath := g.VFioPath()

		// check if file with this path exists
		if _, err := os.Stat(vfioDevPath); err == nil {
			devs = append(devs, vfioDevPath)
		} else {
			return devs, fmt.Errorf("failed to stat device %s, err: %v", vfioDevPath, err)
		}
	}

	return devs, nil
}
