package ilog

import (
	"sync"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

var (
	m   *sync.Mutex
	cfg *zap.Config
	l   *zap.Logger
)

// Init test logger, all other apps should call init explicitly
func init() {
	m = &sync.Mutex{}
	InitForTests()
}

func doSetCfg(c *zap.Config) {
	if c == nil {
		new := zap.NewDevelopmentConfig()
		c = &new
	}
	cfg = c
}

func doInitLogger() {
	logger, err := cfg.Build()
	if err != nil {
		panic(err)
	}
	l = logger
}

func InitWithConfig(cfg zap.Config) {
	doSetCfg(&cfg)
	doInitLogger()
}

func Init() {
	m.Lock()
	doSetCfg(nil)
	cfg.OutputPaths = []string{"stdout"}
	cfg.Level.SetLevel(zap.InfoLevel)
	doInitLogger()
	m.Unlock()
}

func InitForTests() {
	m.Lock()
	doSetCfg(nil)
	cfg.OutputPaths = []string{"stdout"}
	cfg.ErrorOutputPaths = []string{"stdout"}
	cfg.Level.SetLevel(zap.DebugLevel)
	doInitLogger()
	m.Unlock()
}

func Log() *zap.Logger {
	m.Lock()
	logger := l
	m.Unlock()
	return logger
}

func SetLogger(logger *zap.Logger) {
	m.Lock()
	l = logger
	m.Unlock()
}

func Cfg() *zap.Config {
	m.Lock()
	c := cfg
	m.Unlock()

	return c
}

func SetLevel(l zapcore.Level) {
	m.Lock()
	cfg.Level.SetLevel(l)
	m.Unlock()
}
