package utils

import (
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
)

const (
	NvidiaModulePath = "/sys/module/nvidia/version"
	NidiaUtilsDir    = "/opt/yandex-nvidia-utils"
)

func GetNvidiaModuleVersion() (string, error) {
	var err error
	version := os.Getenv("NVIDIA_MOD_VERSION")
	if version == "" {
		version, err = sysfsRead(NvidiaModulePath)
		if err != nil {
			return "", fmt.Errorf("fail to open file %s, kernel module is not loaded, %w",
				NvidiaModulePath, err)
		}
	}
	return version, nil
}

func getNvidiaUtilsPath(name string) (string, error) {
	version, err := GetNvidiaModuleVersion()
	if err != nil {
		return "", err
	}
	path := filepath.Join(NidiaUtilsDir, version, name)
	_, err = os.Stat(path)
	if err != nil {
		return "", fmt.Errorf("kernel module version %s is not supported, cant find :%s", version, path)
	}
	return path, nil
}

func GetNvidiaLibraryPath() (string, error) {
	return getNvidiaUtilsPath("")
}

func GetNvidiaBinPath(name string) (string, error) {
	return getNvidiaUtilsPath(name)
}

func sysfsRead(p string) (string, error) {
	data, err := ioutil.ReadFile(p)
	if err != nil {
		return "", err
	}
	s := string(data)
	if idx := strings.IndexRune(s, '\n'); idx != -1 {
		s = s[:idx]
	}
	return strings.TrimSpace(s), nil
}
