package utils

import (
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"
	"time"

	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
)

const (
	PeriodicTaskInited = iota
	PeriodicTaskInProgress
	PeriodicTaskFinished
)

type PeriodicTaskInfo struct {
	progName string
	progArgs []string
	progEnv  []string

	cmd *exec.Cmd

	launchPeriod time.Duration
	timeout      time.Duration

	currentStdout string
	currentStderr string

	LastLaunchTime time.Time
	LastStdout     string
	LastStderr     string
	LastStatus     int

	LaunchState    int
	errorChannel   chan error
	startedChannel chan int
}

func NewPeriodicTaskInfo(
	cmd string,
	args []string,
	launchPeriod time.Duration,
	timeout time.Duration,
	env []string) *PeriodicTaskInfo {

	return &PeriodicTaskInfo{
		progName: cmd,
		progArgs: args,
		progEnv:  env,

		cmd: nil,

		launchPeriod: launchPeriod,
		timeout:      timeout,

		currentStdout: "",
		currentStderr: "",

		LastLaunchTime: time.Now(),
		LastStdout:     "",
		LastStderr:     "",
		LastStatus:     0,

		LaunchState:    PeriodicTaskInited,
		errorChannel:   make(chan error, 1),
		startedChannel: make(chan int, 1)}
}

func (r *PeriodicTaskInfo) GetCommandStr() string {
	if r == nil {
		return ""
	}

	result := r.progName
	for _, s := range r.progArgs {
		if strings.Contains(s, " ") {
			s = "\"" + s + "\""
		}
		result = result + " " + s
	}

	return result
}

func (r *PeriodicTaskInfo) Update() error {

	launch := func(cmd *exec.Cmd, stdout *string, stderr *string, c chan error, s chan int) {

		var outPipe, errPipe io.ReadCloser
		var err error

		outPipe, err = cmd.StdoutPipe()
		if err != nil {
			c <- err
			return
		}

		errPipe, err = cmd.StderrPipe()
		if err != nil {
			c <- err
			return
		}

		ilog.Log().Debug("starting Periodic Task")
		err = cmd.Start()
		if err != nil {
			c <- err
			ilog.Log().Debug("start of Periodic Task failed", zap.Error(err))
			return
		}
		ilog.Log().Debug("Periodic Task started successfully")

		s <- 0

		outBytes, _ := ioutil.ReadAll(outPipe)
		*stdout = string(outBytes)

		errBytes, _ := ioutil.ReadAll(errPipe)
		*stderr = string(errBytes)

		c <- cmd.Wait()
	}

	switch r.LaunchState {

	case PeriodicTaskInited:
		r.cmd = exec.Command(r.progName, r.progArgs...)
		r.cmd.Env = append(os.Environ(), r.progEnv...)

		go launch(r.cmd, &r.currentStdout, &r.currentStderr, r.errorChannel, r.startedChannel)
		<-r.startedChannel

		r.LastLaunchTime = time.Now()
		r.LaunchState = PeriodicTaskInProgress

		return nil

	case PeriodicTaskInProgress:
		if len(r.errorChannel) != 0 {
			err := <-r.errorChannel
			if err != nil {
				switch err.(type) {
				case *exec.ExitError:

				default:
					return err
				}
			}

			r.LastStatus = r.cmd.ProcessState.ExitCode()
			r.LastStderr = r.currentStderr
			r.LastStdout = r.currentStdout
			r.LaunchState = PeriodicTaskFinished

			ilog.Log().Debug("Periodic Task finished")

		} else {
			if r.timeout == 0*time.Second {
				return nil
			}
			if time.Now().After(r.LastLaunchTime.Add(r.timeout)) {
				err := r.cmd.Process.Kill()
				if err != nil {
					return err
				}

				r.LastStderr = "killed by timeout"
				r.LastStdout = ""
				r.LastStatus = 1
				r.LaunchState = PeriodicTaskFinished

				ilog.Log().Debug("Periodic Task was killed by timeout")
			}
		}

	case PeriodicTaskFinished:
		currentTime := time.Now()
		if currentTime.Before(r.LastLaunchTime.Add(r.launchPeriod)) {
			return nil
		}
		r.cmd = exec.Command(r.progName, r.progArgs...)
		r.cmd.Env = append(os.Environ(), r.progEnv...)

		go launch(r.cmd, &r.currentStdout, &r.currentStderr, r.errorChannel, r.startedChannel)
		<-r.startedChannel

		r.LastLaunchTime = currentTime
		r.LaunchState = PeriodicTaskInProgress

		return nil
	}
	return nil
}
