package utils

import (
	"testing"
	"time"
)

func TestGetCommandStr(t *testing.T) {
	task := NewPeriodicTaskInfo("echo", []string{"-ne", "test"}, time.Second*2, time.Second*0, []string{})
	cmdStr := task.GetCommandStr()

	cmdStrRef := "echo -ne test"

	if cmdStr != cmdStrRef {
		t.Errorf("Wrong command string: got '%s', should be '%s'", cmdStr, cmdStrRef)
	}

	task = NewPeriodicTaskInfo("bash", []string{"-c", "echo -ne test"}, time.Second*2, time.Second*0, []string{})
	cmdStr = task.GetCommandStr()

	cmdStrRef = "bash -c \"echo -ne test\""

	if cmdStr != cmdStrRef {
		t.Errorf("Wrong command string: got '%s', should be '%s'", cmdStr, cmdStrRef)
	}
}

func TestUpdate(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "echo -ne test"}, time.Second*2, time.Second*0, []string{})

	stateRef := PeriodicTaskInited

	if task.LaunchState != stateRef {
		t.Errorf("PeriodicTask was inited with error: init state is %d, should be %d", task.LaunchState, stateRef)
		return
	}

	err := task.Update()
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	stateRef = PeriodicTaskInProgress

	if task.LaunchState != stateRef {
		t.Errorf("PeriodicTask has wrong state after first Update: state is %d, should be %d", task.LaunchState, stateRef)
		return
	}

	<-time.NewTimer(time.Second).C

	err = task.Update()
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	stateRef = PeriodicTaskFinished

	if task.LaunchState != stateRef {
		t.Errorf("PeriodicTask has wrong state after second of work: state is %d,  waiting for %d", task.LaunchState, stateRef)
		return
	}

	<-time.NewTimer(time.Second * 2).C

	err = task.Update()
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	stateRef = PeriodicTaskInProgress

	if task.LaunchState != stateRef {
		t.Errorf("PeriodicTask has wrong state after 3 seconds of waiting: state is %d,  waiting for %d", task.LaunchState, stateRef)
	}
}

func TestNumOfLaunches(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "echo test"}, time.Second*1, time.Second*0, []string{})
	timer := time.NewTimer(time.Second*10 - time.Millisecond*200)
	num := 0

	lastLaucnhedTime := task.LastLaunchTime

	for range time.Tick(time.Millisecond) {
		err := task.Update()
		if err != nil {
			t.Errorf("Error on Update() call: %s", err.Error())
		}
		if task.LaunchState == PeriodicTaskInProgress && task.LastLaunchTime != lastLaucnhedTime {
			num += 1
			lastLaucnhedTime = task.LastLaunchTime
		}
		if len(timer.C) != 0 {
			break
		}
	}

	numRef := 10

	if num != numRef {
		t.Errorf("Wrong number of launches in 10 seconds: got number is %d, waiting for %d", num, numRef)
	}
}

func launchAndWait(task *PeriodicTaskInfo, tick time.Duration) error {
	for range time.Tick(tick) {
		err := task.Update()
		if err != nil {
			return err
		}
		if task.LaunchState == PeriodicTaskFinished {
			return nil
		}
	}
	return nil
}

func TestLaunchWithoutError(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "test 42 -eq 42"}, time.Second*2, time.Second*0, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	statusRef := 0

	if task.LastStatus != 0 {
		t.Errorf("Wrong exit code of '%s' command: got %d, waiting for %d", task.GetCommandStr(), task.LastStatus, statusRef)
	}
}

func TestLaunchWithError(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "test 42 -eq 4"}, time.Second*2, time.Second*0, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	statusRef := 1

	if task.LastStatus != statusRef {
		t.Errorf("Wrong exit code of '%s' command: got %d, waiting for %d", task.GetCommandStr(), task.LastStatus, statusRef)
	}
}

func TestStdout(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "echo -ne test"}, time.Second*2, time.Second*0, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: '%s'", err.Error())
	}

	stdoutRef := "test"

	if task.LastStdout != stdoutRef {
		t.Errorf("Wrong stdout of '%s' command: got '%s', waiting for '%s'", task.GetCommandStr(), task.LastStdout, stdoutRef)
	}
}

func TestStderr(t *testing.T) {
	task := NewPeriodicTaskInfo("bash", []string{"-c", "cat --qwerty"}, time.Second*2, time.Second*0, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	stderrRef := "cat: unrecognized option '--qwerty'\nTry 'cat --help' for more information.\n"

	if task.LastStderr != stderrRef {
		t.Errorf("Wrong stderr of '%s' command: got \n%s\n, waiting for\n%s", task.GetCommandStr(), task.LastStderr, stderrRef)
	}
}

func TestLaunchWithoutTimeout(t *testing.T) {
	timeLimit := time.Millisecond * 200
	task := NewPeriodicTaskInfo("sleep", []string{"0.1"}, time.Second*2, timeLimit, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	statusRef := 0

	if task.LastStatus != statusRef {
		t.Errorf("Wrong status on '%s' command with downtime %d ms, got %d, waiting for %d", task.GetCommandStr(), timeLimit.Milliseconds(), task.LastStatus, statusRef)
	}
}

func TestLaunchWithTimeout(t *testing.T) {
	timeLimit := time.Millisecond * 500
	task := NewPeriodicTaskInfo("sleep", []string{"4"}, time.Second*2, timeLimit, []string{})
	err := launchAndWait(task, time.Millisecond)
	if err != nil {
		t.Errorf("Error on Update() call: %s", err.Error())
	}

	statusRef := 1
	stdoutRef := ""
	stderrRef := "killed by timeout"

	if task.LastStatus != statusRef {
		t.Errorf("Wrong status on '%s' command with downtime %d ms, got %d, waiting for %d", task.GetCommandStr(), timeLimit.Milliseconds(), task.LastStatus, statusRef)
	}

	if task.LastStdout != stdoutRef {
		t.Errorf("Wrong stdout on '%s' command with downtime %d ms, got %s, waiting for %s", task.GetCommandStr(), timeLimit.Milliseconds(), task.LastStdout, stdoutRef)
	}

	if task.LastStderr != stderrRef {
		t.Errorf("Wrong stderr on '%s' command with downtime %d ms, got %s, waiting for %s", task.GetCommandStr(), timeLimit.Milliseconds(), task.LastStderr, stderrRef)
	}
}
