package utils

import (
	"fmt"
	"io/ioutil"
	"strconv"
	"strings"
)

const ProcDevFile = "/proc/devices"

func ProcGetDevices() (map[string]int, error) {
	data, err := ioutil.ReadFile(ProcDevFile)
	if err != nil {
		return nil, fmt.Errorf("fail to read %s, err: %v", ProcDevFile, err)
	}

	return ParseProcDev(string(data)), nil
}

func ParseProcDev(procDevData string) map[string]int {
	devMap := make(map[string]int)

	for _, line := range strings.Split(procDevData, "\n") {
		if line == "" || line == "Character devices:" || line == "Block devices:" {
			continue
		}

		line = strings.TrimSpace(line)

		majDev := strings.Split(line, " ")
		if len(majDev) != 2 || len(majDev[0]) == 0 || len(majDev[1]) == 0 {
			continue
		}

		maj, err := strconv.Atoi(majDev[0])
		if err != nil {
			continue
		}

		devMap[majDev[1]] = maj
	}

	return devMap
}

func GetDevMaj(dev string) (int, error) {
	devMap, err := ProcGetDevices()
	if err != nil {
		return -1, fmt.Errorf("ProcGetDevices() failed, err: %v", err.Error())
	}

	maj, ok := devMap[dev]
	if !ok {
		return -1, fmt.Errorf("no major number found for device '%s'", dev)
	}

	return maj, nil
}
