package utils

import (
	"github.com/stretchr/testify/require"
	"testing"
)

func TestFabricWithSimpleDataObject(t *testing.T) {
	type TestObj struct {
		TestFieldA int
		TestFieldB float32
		TestFieldC string
		TestFieldD bool
		testFieldF int
	}

	baseObj := TestObj{42, 3.14, "Hello world", false, 24}

	newAVal := 41
	newBVal := float32(2.14)
	newCVal := "goodbye"
	newDVal := true

	values := []map[string]interface{}{
		{},
		{"TestFieldA": newAVal},
		{"TestFieldB": newBVal},
		{"TestFieldC": newCVal},
		{"TestFieldD": newDVal},
		{"TestFieldA": newAVal, "TestFieldB": newBVal, "TestFieldC": newCVal, "TestFieldD": newDVal},
	}

	objs, err := GenerateObjects(baseObj, values)
	require.NoError(t, err)

	require.Equal(t, len(objs), len(values))

	obj := &objs[0]
	require.Equal(t, obj.TestFieldA, baseObj.TestFieldA)
	require.Equal(t, obj.TestFieldB, baseObj.TestFieldB)
	require.Equal(t, obj.TestFieldC, baseObj.TestFieldC)
	require.Equal(t, obj.TestFieldD, baseObj.TestFieldD)

	obj = &objs[1]
	require.Equal(t, obj.TestFieldA, newAVal)
	require.Equal(t, obj.TestFieldB, baseObj.TestFieldB)
	require.Equal(t, obj.TestFieldC, baseObj.TestFieldC)
	require.Equal(t, obj.TestFieldD, baseObj.TestFieldD)

	obj = &objs[2]
	require.Equal(t, obj.TestFieldA, baseObj.TestFieldA)
	require.Equal(t, obj.TestFieldB, newBVal)
	require.Equal(t, obj.TestFieldC, baseObj.TestFieldC)
	require.Equal(t, obj.TestFieldD, baseObj.TestFieldD)

	obj = &objs[3]
	require.Equal(t, obj.TestFieldA, baseObj.TestFieldA)
	require.Equal(t, obj.TestFieldB, baseObj.TestFieldB)
	require.Equal(t, obj.TestFieldC, newCVal)
	require.Equal(t, obj.TestFieldD, baseObj.TestFieldD)

	obj = &objs[4]
	require.Equal(t, obj.TestFieldA, baseObj.TestFieldA)
	require.Equal(t, obj.TestFieldB, baseObj.TestFieldB)
	require.Equal(t, obj.TestFieldC, baseObj.TestFieldC)
	require.Equal(t, obj.TestFieldD, newDVal)

	obj = &objs[5]
	require.Equal(t, obj.TestFieldA, newAVal)
	require.Equal(t, obj.TestFieldB, newBVal)
	require.Equal(t, obj.TestFieldC, newCVal)
	require.Equal(t, obj.TestFieldD, newDVal)

}

func TestFabricWithSimpleDataObjectWithError(t *testing.T) {
	type TestObj struct {
		TestFieldA int
		TestFieldB float32
		TestFieldC string
		TestFieldD bool
		testFieldF int
	}

	baseObj := TestObj{42, 3.14, "Hello world", false, 24}

	objs, err := GenerateObjects(baseObj, []map[string]interface{}{{"testFieldF": 23}})
	require.Error(t, err)
	require.Nil(t, objs)

	objs, err = GenerateObjects(baseObj, []map[string]interface{}{{"TestFieldG": 23}})
	require.Error(t, err)
	require.Nil(t, objs)

	objs, err = GenerateObjects(baseObj, []map[string]interface{}{{"TestFieldA": "error"}})
	require.Error(t, err)
	require.Nil(t, objs)

	objs, err = GenerateObjects(baseObj, []map[string]interface{}{{}, {"testFieldF": 23}})
	require.Error(t, err)
	require.Nil(t, objs)
}

func TestFabricWithComplexObjects(t *testing.T) {

	type TestSimpleObject struct {
		TestFieldC int
		TestFieldD string
	}

	type TestComplexObj struct {
		TestFieldA int
		TestFieldB TestSimpleObject
	}

	baseObj := TestComplexObj{42, TestSimpleObject{24, "Hello"}}

	newAVal := 41
	newCVal := 23
	newDVal := "goodbye"

	values := []map[string]interface{}{
		{},
		{"TestFieldA": newAVal, "TestFieldB": map[string]interface{}{"TestFieldC": newCVal, "TestFieldD": newDVal}},
	}

	objs, err := GenerateObjects(baseObj, values)
	require.NoError(t, err)

	require.Equal(t, len(objs), len(values))

	obj := &objs[0]
	require.Equal(t, obj.TestFieldA, baseObj.TestFieldA)
	require.Equal(t, obj.TestFieldB.TestFieldC, baseObj.TestFieldB.TestFieldC)
	require.Equal(t, obj.TestFieldB.TestFieldD, baseObj.TestFieldB.TestFieldD)

	obj = &objs[1]
	require.Equal(t, obj.TestFieldA, newAVal)
	require.Equal(t, obj.TestFieldB.TestFieldC, newCVal)
	require.Equal(t, obj.TestFieldB.TestFieldD, newDVal)
}
