// Juggler client push client package
package juggler

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"go.uber.org/zap"
)

type JugglerEvent struct {
	Description string `json:"description"`
	Service     string `json:"service"`
	Status      string `json:"status"`
}

type JugglerRequest struct {
	Source string         `json:"source"`
	Events []JugglerEvent `json:"events"`
}

type JugglerReply struct {
	Status  bool   `json:"success"`
	Message string `json:"message"`
	Events  []struct {
		Code  int    `json:"code"`
		Error string `json:"error"`
	}
	AcceptedEvents int `json:"accepted_events"`
}

const (
	DefaultLocalURL = "http://localhost:31579"
)

type JugglerClient struct {
	ServerURL string
	Client    *http.Client
	Header    map[string]string
}

func NewJugglerClientWithURL(serverURL string) (client *JugglerClient) {

	return &JugglerClient{
		ServerURL: serverURL,
		Client: &http.Client{
			Timeout: time.Duration(5 * time.Second),
		},
		Header: make(map[string]string),
	}
}

// NewJugglerClient create client which push events to local juggler service
func NewJugglerClient() (client *JugglerClient) {
	return NewJugglerClientWithURL(DefaultLocalURL)
}

// SetHeader add heades which later will be appended to events on push
func (c *JugglerClient) SetHeader(key string, value string) {
	c.Header[key] = value
}

func (c *JugglerClient) makeRequest(ctx context.Context, json bool, requestBody string) ([]byte, error) {
	req, err := http.NewRequest("POST", c.ServerURL+"/events", strings.NewReader(requestBody))
	if err != nil {
		return nil, err
	}

	for k, v := range c.Header {
		req.Header.Set(k, v)
	}
	if json {
		req.Header.Set("Content-Type", "application/json")
	}
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()
	reply, err := ioutil.ReadAll(resp.Body)
	return reply, err
}

// PushEvents submit events to juggler service
func (c *JugglerClient) PushEvents(ctx context.Context, events JugglerRequest) error {
	eventCount := len(events.Events)
	requestBody, err := json.Marshal(&events)
	if err != nil {
		return err
	}

	responseBody, err := c.makeRequest(ctx, true, string(requestBody))
	if err != nil {
		return err
	}

	var result JugglerReply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return err
	}

	if !result.Status {
		return fmt.Errorf("bad push status: %s", result.Message)
	}

	if result.AcceptedEvents != eventCount {
		var badEvents []string
		for idx, data := range result.Events {
			if data.Error != "" {
				badEvents = append(badEvents, events.Events[idx].Service+":"+data.Error)
			}
		}
		return fmt.Errorf("some events were not accepted: %s", strings.Join(badEvents[:], ","))
	}

	return nil
}

// Examples:
//	c := juggler.NewJugglerClient()
//	c.SetHeader("User-Agent", "my-service-name/0.1")
//	ctx := context.Background()
//	r := juggler.JugglerRequest{
//		Source: "my-service-name",
//		Events: []juggler.JugglerEvent{
//			juggler.JugglerEvent{
//				Description: "My event description",
//				Service:     "my-service-name",
//				Status:      "OK",
//			},
//		},
//	}
//	err := c.PushEvents(ctx, r)

// PushEventLoop read data from channel and push it to jugglerClient
func (c *JugglerClient) PushEventLoop(ctx context.Context, log *zap.Logger, in chan JugglerRequest) {
	dropped := 0
	ts := time.Now()

	for {
		select {
		// If context was cancelled, we stop loop immediately
		case <-ctx.Done():
			return
		case e, more := <-in:
			err := c.PushEvents(ctx, e)
			if err == nil {
				continue
			}
			// Even if we fail to push event, it is not fatal and it is not good
			// idea to log each error, log only once a hour
			if log != nil && dropped > 0 && time.Since(ts) > time.Hour {
				log.Error("push failed",
					zap.Int("errors_dropped", dropped),
					zap.Error(err))
				dropped = 0
				ts = time.Now()
			} else {
				dropped++

			}
			// If channel was closed and we have receiver all data  then we are done
			if !more {
				return
			}
		}
	}
}

// Examples:
//     // This example demonstrate how to submit events w/o blocking main thread
//     ch = make(chan juggler.JugglerRequest, 10),
//     c := juggler.NewJugglerClient()
//     ctx, cancel := context.WithCancel(context.Background())
//     //defer cancel()
//     go c.PushEventLoop(ctx, nil, ch)
//     // Now one can send envents w/o blocking
//	r := juggler.JugglerRequest{
//		Source: "my-service-name",
//		Events: []juggler.JugglerEvent{
//			juggler.JugglerEvent{
//				Description: "My event description",
//				Service:     "my-service-name",
//				Status:      "OK",
//			},
//		},
//	}
//     // Submit event w/o blocking
//     select {
//     case s.jugglerQueue <- e:
//     default:
//          fmtPrint("juggler queue overflow, drop event")
//      }
//     // Later one can close submission channel to signal submitter to stop the loop
//     close(ch)
