package juggler_test

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"os/exec"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/juggler"
)

func startJuggler(ctx context.Context, port string) error {
	var res *http.Response
	cmd := exec.CommandContext(ctx, "juggler-client",
		"--log-level", "debug",
		"--run-dir", "/tmp/",
		"--instance-port", port)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stdout

	err := cmd.Start()
	if err != nil {
		return err
	}
	// Wait for server to start
	for tries := 10; tries != 0; tries-- {
		res, err = http.Get("http://localhost:" + port + "/ping")
		if err == nil && res.StatusCode == 200 {
			break
		}
		if err == nil {
			err = fmt.Errorf("http.Get failed code:%d", res.StatusCode)
		}
		time.Sleep(time.Second)
	}
	return err
}

func TestPushEvent(t *testing.T) {
	var tests = []struct {
		name   string
		events []juggler.JugglerEvent
		err    error
	}{
		{
			name: "single",
			events: []juggler.JugglerEvent{
				juggler.JugglerEvent{
					Description: "e1",
					Service:     "test-sevice",
					Status:      "OK",
				},
			},
			err: nil,
		},
		{
			name: "multiple",
			events: []juggler.JugglerEvent{
				juggler.JugglerEvent{
					Description: "e1",
					Service:     "test-sevice",
					Status:      "OK",
				},
				juggler.JugglerEvent{
					Description: "e1",
					Service:     "test-sevice",
					Status:      "CRIT",
				},
				juggler.JugglerEvent{
					Description: "e2",
					Service:     "test-sevice",
					Status:      "OK",
				},
			},
			err: nil,
		},
		{
			name: "goodAndBad",
			events: []juggler.JugglerEvent{
				juggler.JugglerEvent{
					Description: "goodEvent",
					Service:     "test-sevice",
					Status:      "OK",
				},
				juggler.JugglerEvent{
					Description: "badEvent",
					Service:     "test-sevice",
					Status:      "bad status",
				},
				juggler.JugglerEvent{
					Description: "goodEvent2",
					Service:     "test-sevice",
					Status:      "OK",
				},
			},
			err: fmt.Errorf("some events were not accepted: test-sevice:Status \"BAD STATUS\" is not valid, must be one of ('OK', 'INFO', 'WARN', 'CRIT')"),
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	port := "31581"
	err := startJuggler(ctx, port)
	require.NoError(t, err)

	c := juggler.NewJugglerClientWithURL("http://localhost:" + port)
	c.SetHeader("User-Agent", "test-service/1.0")

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			req := juggler.JugglerRequest{
				Source: "test-service",
				Events: tt.events,
			}
			err := c.PushEvents(ctx, req)
			require.Equal(t, tt.err, err)
		})
	}
}

func TestPushEventLoop(t *testing.T) {
	var events []juggler.JugglerEvent

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	port := "31582"
	err := startJuggler(ctx, port)
	require.NoError(t, err)
	c := juggler.NewJugglerClientWithURL("http://localhost:" + port)

	ch := make(chan juggler.JugglerRequest, 2)
	go c.PushEventLoop(ctx, ilog.Log(), ch)

	for i := 0; i < 50; i++ {
		e := juggler.JugglerEvent{
			Description: fmt.Sprintf("event-%d", i),
			Service:     "test-sevice",
			Status:      "OK",
		}
		events = append(events, e)
		ch <- juggler.JugglerRequest{
			Source: "test-service",
			Events: events,
		}
		time.Sleep(time.Millisecond * 10)
	}
	close(ch)
}
