package modprobe

import (
	"bytes"
	"context"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"go.uber.org/zap"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/internal/ilog"
)

const (
	ConfigDir = "/etc/modprobe.d"
)

type Module struct {
	Names   []string
	Alias   string
	Options []string
}

func run(ctx context.Context, cmd string, arg ...string) (string, string, error) {
	c := exec.CommandContext(ctx, cmd, arg...)
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	c.Stdout = stdout
	c.Stderr = stderr
	err := c.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", c.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))
	return stdout.String(), stderr.String(), err

}

// LoadedModules queries the running kernel's modules.
func LoadedModules() []string {
	moduleDirs, err := ioutil.ReadDir("/sys/module")
	if err != nil {
		ilog.Log().Error("Could not fetch loaded kernel modules", zap.Error(err))
		return []string{}
	}

	var modules []string

	for _, moduleDir := range moduleDirs {
		module := moduleDir.Name()
		modules = append(modules, module)
	}

	return modules
}

// IsModuleLoaded tells whether given kernel module is loaded or not.
func IsModuleLoaded(searchedModule string) bool {
	modules := LoadedModules()
	s := strings.ReplaceAll(searchedModule, "-", "_")
	for _, module := range modules {
		if module == s {
			return true
		}
	}

	return false
}

// LoadModule loads a kernel module (via modprobe).
func LoadModule(module string) error {
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)

	cmd := exec.Command("modprobe", module)
	cmd.Stdout = stdout
	cmd.Stderr = stderr
	err := cmd.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", cmd.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))

	return err
}

// LoadModuleIfUnloaded loads a kernel module if it is currently not loaded.
func LoadModuleIfUnloaded(module string) error {
	if !IsModuleLoaded(module) {
		return LoadModule(module)
	}
	return nil
}

// UnloadModule unloads a kernel module (via rmmod).
func UnloadModule(module string) error {
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)

	cmd := exec.Command("rmmod", module)
	cmd.Stdout = stdout
	cmd.Stderr = stderr
	err := cmd.Run()
	ilog.Log().Info("exec",
		zap.String("cmd", cmd.String()),
		zap.Error(err),
		zap.ByteString("stdout", stdout.Bytes()),
		zap.ByteString("stderr", stderr.Bytes()))

	return err
}

// GenConfigPath returns config file path for given entry
func GenConfigPath(entry string) string {
	return filepath.Join(ConfigDir, entry+".conf")
}

// BlacklistModule add module to blacklist file
func BlacklistModule(module string, fname string) error {

	f, err := os.OpenFile(fname, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		return err
	}
	defer f.Close()
	str := "# Completely blacklist " + module + "\n"
	str += "blacklist " + module + "\n"
	str += "install " + module + " /bin/false\n"
	_, err = f.WriteString(str)

	return err
}
