package yasm

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

const (
	DefaultLocalURL = "http://localhost:11005"
)

var (
	ErrSendFailed  = errors.New("fail to send request")
	ErrBadResponse = errors.New("bad reponse status status")
)

type YasmValue struct {
	Name  string      `json:"name"`
	Value interface{} `json:"val"`
}

type YasmMetrics struct {
	Tags   map[string]string `json:"tags"`
	TTL    uint              `json:"ttl"`
	Values []YasmValue       `json:"values"`
}

type YasmReply struct {
	Status      string `json:"status"`
	ErrorString string `json:"error"`
}

type YasmClient struct {
	ServerURL string
	Client    *http.Client
	Header    map[string]string
}

func NewYasmClientWithURL(serverURL string) (client *YasmClient) {

	return &YasmClient{
		ServerURL: serverURL,
		Client: &http.Client{
			Timeout: time.Duration(5 * time.Second),
		},
		Header: make(map[string]string),
	}
}

// NewYasmClient create client which push events to local yasm service
func NewYasmClient() (client *YasmClient) {
	return NewYasmClientWithURL(DefaultLocalURL)
}

// SetHeader add heades which later will be appended to events on push
func (c *YasmClient) SetHeader(key string, value string) {
	c.Header[key] = value
}

func (c *YasmClient) makeRequest(ctx context.Context, json bool, requestBody string) ([]byte, error) {
	req, err := http.NewRequest("POST", c.ServerURL, strings.NewReader(requestBody))
	if err != nil {
		return nil, err
	}

	for k, v := range c.Header {
		req.Header.Set(k, v)
	}
	if json {
		req.Header.Set("Content-Type", "application/json")
	}
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()
	reply, err := ioutil.ReadAll(resp.Body)
	return reply, err
}

func (c *YasmClient) SendMetrics(ctx context.Context, metrics []YasmMetrics) error {
	requestBody, err := json.Marshal(&metrics)
	if err != nil {
		return fmt.Errorf("bad request %w err:%s", ErrSendFailed, err.Error())
	}

	responseBody, err := c.makeRequest(ctx, true, string(requestBody))
	if err != nil {
		return fmt.Errorf("submit failed %w err:%s", ErrSendFailed, err.Error())
	}

	var result YasmReply
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return fmt.Errorf("fail to parse request %w, err:%s", ErrSendFailed, err.Error())
	}

	if result.Status != "ok" {
		return fmt.Errorf("fail to send metrics, staus:%s errmsg: %s %w", result.Status, result.ErrorString, ErrBadResponse)
	}

	return nil
}

// Probe try to push probe event, exit if push was successfull or timeout reached
func (c *YasmClient) Probe(ctx context.Context, timeout time.Duration) error {
	ctx, cancel := context.WithTimeout(ctx, timeout)
	defer cancel()

	for {
		m := YasmMetrics{
			Tags: map[string]string{"itype": "probe"},
			TTL:  30,
			Values: []YasmValue{
				YasmValue{
					Name:  "probe_xxxx",
					Value: 1,
				},
			},
		}
		err := c.SendMetrics(ctx, []YasmMetrics{m})
		if err == nil {
			return nil
		}
		select {
		case <-ctx.Done():
			return errors.New("context canceled")
		default:
			time.Sleep(time.Millisecond * 100)
		}
	}
}
