package yasm_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"a.yandex-team.ru/infra/rsm/nvgpumanager/pkg/yasm"
)

func TestPushEvent(t *testing.T) {
	var tests = []struct {
		name    string
		metrics yasm.YasmMetrics
		err     error
	}{
		{
			name: "single",
			metrics: yasm.YasmMetrics{
				Tags: map[string]string{"itype": "runtimecloud"},
				Values: []yasm.YasmValue{
					yasm.YasmValue{
						Name:  "test-service-val_count_tmmv",
						Value: 1234,
					},
				},
			},
			err: nil,
		},
		{
			name: "multi",
			metrics: yasm.YasmMetrics{
				Tags: map[string]string{"itype": "runtimecloud"},
				Values: []yasm.YasmValue{
					yasm.YasmValue{
						Name:  "test-service-val1_tmmv",
						Value: 1,
					},
					yasm.YasmValue{
						Name:  "test-service-val2_tmmv",
						Value: 100,
					},
					yasm.YasmValue{
						Name:  "test-service-val3_tmmv",
						Value: 10000,
					},
				},
			},
			err: nil,
		}, {
			name: "bad-metrics",
			metrics: yasm.YasmMetrics{
				Tags: map[string]string{"itype": "runtimecloud"},
				Values: []yasm.YasmValue{
					yasm.YasmValue{
						Name:  "test-service-badvalwosuffix",
						Value: "bad",
					},
				},
			},
			err: yasm.ErrSendFailed,
		},
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	c := yasm.NewYasmClient()
	c.SetHeader("User-Agent", "test-service/1.0")

	err := c.Probe(ctx, 20*time.Second)
	require.NoErrorf(t, err, "yasm response timeout")

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := []yasm.YasmMetrics{tt.metrics}
			err := c.SendMetrics(ctx, m)
			if tt.err == nil {
				require.NoError(t, err)
			} else {
				require.True(t, true, errors.Is(err, tt.err))
			}
		})
	}
}
