#include "error.h"
#include "tests.cuh"

#include <library/cpp/svnversion/svnversion.h>

#include <cuda_runtime.h>

#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/file.h>
#include <sys/select.h>
#include <sys/wait.h>


#define MIN_ELEMS_NUMB 8192
#define DEFAULT_ELEMS_NUMB (100 * MB)

#define MIN_BUF_SIZE (MIN_ELEMS_NUMB * sizeof(MIN_ELEMS_NUMB))
#define DEFAULT_BUF_SIZE (DEFAULT_ELEMS_NUMB * sizeof(MIN_ELEMS_NUMB))

#define DEFINE_TEST_DESC(test_name) {#test_name, (void *)test_name, 0, 0}
#define DEFINE_PERF_TEST_DESC(test_name) {#test_name, (void *)test_name, 1, 0}
#define DEFINE_TEST_LONG_OPT(test_name) {#test_name, no_argument, &(tests_table[NR_##test_name].is_requested), 1}

#define is_perf_test(test_idx) tests_table[test_idx].is_perf_test

#define DEFAULT_COMPLETION_TIMEOUT_SEC (5 * 60)
#define DEFAULT_KILL_TIMEOUT_SEC 5

#define LOCK_FILE_PATH "/tmp/cuda-check.lock"


typedef cudaError_t (*test_cb)(size_t elems_numb, ssize_t gpu_idx, int gpu_count, int run_on_all_gpus);
typedef cudaError_t (*perf_test_cb)(size_t elems_numb, ssize_t gpu_idx, int gpu_count, int run_on_all_gpus, ssize_t threshold);


size_t parse_cmdline_args(int argc, char* argv[], ssize_t *gpu_idx, size_t *elems_numb, ssize_t *threshold, int *is_yt_check, int *is_manual_threshold, ssize_t *completion_timeout, ssize_t *kill_timeout, int *show_gpu_count);
void print_help(const char *prog_name);
void print_tests_list();
int parse_size(const char *optarg, ssize_t *parsed_size);
int parse_time(const char *optarg, ssize_t *parsed_time);


struct test_desc {
	const char *name;
	void *callback;
	int is_perf_test;
	int is_requested;
};

static struct test_desc tests_table[] {
	// functional tests
	DEFINE_TEST_DESC(test_memcpy),
	DEFINE_TEST_DESC(test_saxpy_unified),
	DEFINE_TEST_DESC(test_host_async),
	DEFINE_TEST_DESC(test_host_async_batched),
	DEFINE_TEST_DESC(test_p2p),
	DEFINE_TEST_DESC(test_dev_sync),

	// performance tests - need to specify test explicitly to run
	DEFINE_PERF_TEST_DESC(test_p2p_bandwidth),
	DEFINE_PERF_TEST_DESC(test_p2p_bandwidth_p2p_off),
	DEFINE_PERF_TEST_DESC(test_p2p_bandwidth_kernel),

	// add new 'test_*' above
};

enum gpu_threshold_numbs {
	NR_threshold_a100_40g,
	NR_threshold_a100_80g,

	// add new GPUs thresholds above

	TOTAL_GPU_THRESHOLDS_COUNT
};

struct gpu_threshold_desc {
	const char *name;
	ssize_t threshold;
};

// set minimal threshold as ~90% of minimal bandwidth seen in 'test_p2p_bandwidth'/'test_p2p_bandwidth_kernel' tests for these GPUs
static struct gpu_threshold_desc gpu_thresholds_table[] {
	{"A100-SXM4-40GB", 202 * GB},
	{"A100-SXM-80GB",  219 * GB},

	// add new GPUs thresholds above
};

size_t __count_requested_tests()
{
	size_t requested_tests_nr = 0;

	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++)
		requested_tests_nr += tests_table[i].is_requested;

	return requested_tests_nr;
}

size_t __invert_requested_tests_except_perf()
{
	size_t requested_tests_nr = 0;

	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++)
		if (!is_perf_test(i)) {
			tests_table[i].is_requested = !(tests_table[i].is_requested);
			requested_tests_nr += tests_table[i].is_requested;
		}

	return requested_tests_nr;
}

size_t __request_tests(int func, int perf)
{
	size_t requested_tests_nr = 0;

	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++)
		if ((func && !is_perf_test(i)) ||
		    (perf && is_perf_test(i))) {
			tests_table[i].is_requested = 1;
			requested_tests_nr++;
		}

	return requested_tests_nr;
}

size_t __request_all_func_tests()
{
	return __request_tests(1, 0);
}

size_t __request_all_perf_tests()
{
	return __request_tests(0, 1);
}

size_t __request_all_non_perf_tests()
{
	// TODO: add new test types here
	return __request_all_func_tests();
}

// returns 0 on success, -1 if test with such idx wasn't found
size_t __unrequest_test(size_t test_idx)
{
	if (test_idx > TOTAL_TESTS_COUNT)
		return -1;

	tests_table[test_idx].is_requested = 0;
	return 0;
}

int main(int argc, char* argv[])
{
	size_t requested_tests_nr = 0;
	ssize_t gpu_idx = -1;
	int run_on_all_gpus = 1;
	size_t elems_numb = DEFAULT_ELEMS_NUMB;
	ssize_t threshold = DEFAULT_THRESHOLD;
	int is_yt_check = 0;
	int is_manual_threshold = 0;
	ssize_t completion_timeout = DEFAULT_COMPLETION_TIMEOUT_SEC;
	ssize_t kill_timeout = DEFAULT_KILL_TIMEOUT_SEC;
	int show_gpu_count = 0;

	cudaError_t err = cudaSuccess;
	int gpu_count = 0;

	eprintf("cuda-check, revision %s\n", GetArcadiaLastChange());
	eprintf("\n");

	if (argc > 1)
		requested_tests_nr = parse_cmdline_args(argc, argv, &gpu_idx, &elems_numb, &threshold, &is_yt_check, &is_manual_threshold, &completion_timeout, &kill_timeout, &show_gpu_count);
	else
		requested_tests_nr = __request_all_non_perf_tests();

	// If there is a stuck child (cuda-check worker) from previous launch - there is a high chance
	// that our launch will be stuck too. So, to prevent hung children multiplication - fail now.
	//
	// Also, we do not want to allow multiple cuda-checks to be run simultaneously because:
	//  * this can affect checks' performance and lead to exceeding timeouts or ENOMEM errors
	//  * when using in production before actual gpu-jobs execution - detecting that second check
	//    is trying to start is usually a sign of environment misconfiguration as jobs run isolated
	//    from each other with single cuda-check run before gpu-job start
	int lock_fd = open(LOCK_FILE_PATH, O_RDONLY | O_CREAT, S_IRUSR);
	if (lock_fd == -1) {
		eprintf("'open(%s)' failed: %s\n", LOCK_FILE_PATH, strerror(errno));
		exit(EXIT_FAILURE);
	}

	if (flock(lock_fd, LOCK_EX | LOCK_NB)) {
		if (errno == EWOULDBLOCK)
			eprintf("Another instance of cuda-check is running now or stuck cuda-check worker from previous launch is detected: exiting.\n");
		else
			eprintf("'flock()' failed: %s\n", strerror(errno));

		exit(EXIT_FAILURE);
	}
	// If we get here - file is locked and lock will be duplicated to child after fork.
	// Even if child will hang and parent will exit - file will remain locked preventing
	// subsequent launches of cuda-check.

	// connect parent (read end) and child (write end) via pipe for tracking child's lifespan
	int pipefds[2];
	if (pipe2(pipefds, O_NONBLOCK) == -1) {
		eprintf("'pipe2()' failed: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	// run all cuda operations in child aka 'cuda-check worker'
	pid_t child_pid = fork();
	if (child_pid == -1) {
		eprintf("'fork()' failed: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}
	if (child_pid != 0) { // parent (has pipe read fd)
		if (close(pipefds[1])) {
			eprintf("'close(pipefds[1])' failed in parent: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		fd_set rfds;
		struct timeval tv;

		FD_ZERO(&rfds);
		FD_SET(pipefds[0], &rfds);

		tv.tv_sec = completion_timeout;
		tv.tv_usec = 0;

		int ret = select(pipefds[0] + 1, &rfds, NULL, NULL, &tv);
		if (ret == -1) {
			eprintf("'select()' failed: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		} else if (ret == 0) {
			eprintf("\ncuda-check worker (pid=%d) exceeded timeout, let's try to kill it\n", child_pid);
			if (kill(child_pid, SIGKILL)) {
				eprintf("'kill(child_pid=%d)' failed: %s\n", child_pid, strerror(errno));
				exit(EXIT_FAILURE);
			}

			// in order to wait for signal delivery, let's wait a bit in second 'select()'
			FD_ZERO(&rfds);
			FD_SET(pipefds[0], &rfds);

			tv.tv_sec = kill_timeout;
			tv.tv_usec = 0;

			int ret2 = select(pipefds[0] + 1, &rfds, NULL, NULL, &tv);
			if (ret2 == 1) {
				if (FD_ISSET(pipefds[0], &rfds)) {
					eprintf("cuda-check worker is dead now\n");
					// fall through to wait for deceased child
				} else {
					eprintf("second 'select()' unexpectedly returned %d, but the only fd in set is not ready, errno: %s\n", ret2, strerror(errno));
					exit(EXIT_FAILURE);
				}
			} else if (ret2 == 0) {
				eprintf("cuda-check worker (pid=%d) is still alive and supposedly stuck somewhere, exiting with error\n", child_pid);
				exit(EXIT_FAILURE);
			} else {
				eprintf("second 'select()' failed with ret_val=%d and errno: %s\n", ret2, strerror(errno));
				exit(EXIT_FAILURE);
			}
		} else {
			// child finished before timeout
			if (FD_ISSET(pipefds[0], &rfds) != 1) {
				eprintf("'select()' unexpectedly returned %d, but the only fd in set is not ready, errno: %s\n", ret, strerror(errno));
				exit(EXIT_FAILURE);
			}
			// fall through to wait for deceased child
		}

		int wstatus = 0;
		if (waitpid(child_pid, &wstatus, 0) == -1) {
			eprintf("'waitpid()' failed: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		// only here we can unlink lock file (still under lock) as child successfully exited somehow
		if (unlink(LOCK_FILE_PATH)) {
			eprintf("'unlink(%s)' failed: %s\n", LOCK_FILE_PATH, strerror(errno));
			exit(EXIT_FAILURE);
		}
		// 'close()' of last fd will unlock the file lock
		if (close(lock_fd)) {
			eprintf("'close()' failed: %s\n", strerror(errno));
			exit(EXIT_FAILURE);
		}

		if (WIFEXITED(wstatus)) {
			exit(WEXITSTATUS(wstatus));
		} else if (WIFSIGNALED(wstatus)) {
			eprintf("cuda-check worker (pid=%d) was killed by signal %d\n", child_pid, WTERMSIG(wstatus));
			exit(EXIT_FAILURE);
		}

		eprintf("parent should not get here\n");
		exit(EXIT_FAILURE);
	}

	// child (has pipe write fd)
	if (close(pipefds[0])) {
		eprintf("'close(pipefds[0])' failed in child: %s\n", strerror(errno));
		exit(EXIT_FAILURE);
	}

	err = cudaGetDeviceCount(&gpu_count);
	if (err != cudaSuccess) {
		eprintfl("'cudaGetDeviceCount()' failed: %s\n", cudaGetErrorString(err));
		exit(EXIT_FAILURE);
	}

	if (show_gpu_count) {
		eprintf("Detected GPUs count = %d\n", gpu_count);
		exit(EXIT_SUCCESS);
	}

	// =====================================================================
	// move these out of 'parse_cmdline_args()' in order to free parent of any cuda interactions
	// ---------------------------------------------------------------------
	if (gpu_idx > (gpu_count - 1)) {
		eprintf("Invalid gpu index (gpu count = %d)!\n", gpu_count);
		eprintf("\n");
		print_help(argv[0]);
		exit(EXIT_FAILURE);
	}
	// ---------------------------------------------------------------------
	if (is_yt_check) {
		/* assume that YT host has 8 identical GPUs and for known GPUs enable suitable perf_tests
		 * and bandwidth threshold (if no manual threshold has been provided)
		 */
		cudaDeviceProp props;
		err = cudaGetDeviceProperties(&props, 0);
		if (err != cudaSuccess) {
			eprintfl("'cudaGetDeviceProperties()' failed: %s\n", cudaGetErrorString(err));
			exit(EXIT_FAILURE);
		}

		// right now, sizeof(props.name) = 256
		size_t _max_name_len = max(sizeof(props.name), size_t(256));
		for (size_t i = 0; i < TOTAL_GPU_THRESHOLDS_COUNT; i++) {
			if (!strncmp(gpu_thresholds_table[i].name, props.name, _max_name_len)) {
				if (!is_manual_threshold)
					threshold = gpu_thresholds_table[i].threshold;

				printf("Detected GPU model: '%s', using %sthreshold: %zd.\n",
						gpu_thresholds_table[i].name, is_manual_threshold ? "manual " : "", threshold);
				printf("\n");

				// request perf_tests only for GPUs with known threshold
				__request_all_perf_tests();

				// unrequest unnecessary perf test
				int ret = __unrequest_test(NR_test_p2p_bandwidth_p2p_off);
				if (ret != 0) {
					eprintf("'__unrequest_test(%d)' failed: ret=%d\n", NR_test_p2p_bandwidth_p2p_off, ret);
					exit(EXIT_FAILURE);
				}

				requested_tests_nr = __count_requested_tests();

				break;
			}
		}
	}
	// =====================================================================

	if (requested_tests_nr == 0) {
		eprintf("No tests selected!\n");
		printf("\n");
		print_tests_list();
		exit(EXIT_FAILURE);
	}

	if (gpu_idx != -1)
		run_on_all_gpus = 0;

	size_t cur_test_nr = 0;
	eprintf("Start testing (%zu test%s selected, %d gpu%s detected (testing on %d gpu%s)):\n\n", requested_tests_nr,
		                                           (requested_tests_nr == 1) ? "" : "s", gpu_count,
							   (gpu_count) ? "s" : "",
							   (run_on_all_gpus) ? gpu_count : 1,
		                                           (run_on_all_gpus && (gpu_count > 1)) ? "s" : "");
	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++) {
		if (tests_table[i].is_requested == 0)
			continue;

		cur_test_nr++;
		printf("(%zu/%zu) running '%s':\n", cur_test_nr, requested_tests_nr, tests_table[i].name);

		if (!is_perf_test(i))
			err = ((test_cb)tests_table[i].callback)(elems_numb, gpu_idx, gpu_count, run_on_all_gpus);
		else
			err = ((perf_test_cb)tests_table[i].callback)(elems_numb, gpu_idx, gpu_count, run_on_all_gpus, threshold);

		if (err != cudaSuccess) {
			eprintf("(%zu/%zu) '%s' failed with error: %s\n",
			        cur_test_nr, requested_tests_nr,
				tests_table[i].name, cudaGetErrorString(err));
			exit(EXIT_FAILURE);
		}

		printf("(%zu/%zu) '%s' passed\n\n", cur_test_nr, requested_tests_nr, tests_table[i].name);
	}

	eprintf("%zu/%zu test%s passed successfully!\n", cur_test_nr, requested_tests_nr,
			                                (requested_tests_nr == 1) ? "" : "s");
	exit(EXIT_SUCCESS);
}

// as long as this func is called from parent - keep it free from any cuda* calls
size_t parse_cmdline_args(int argc, char* argv[], ssize_t *gpu_idx, size_t *elems_numb, ssize_t *threshold, int *is_yt_check, int *is_manual_threshold, ssize_t *completion_timeout, ssize_t *kill_timeout, int *show_gpu_count)
{
	int err = 0;
	ssize_t buf_size = DEFAULT_BUF_SIZE;

	struct option long_options[] = {
		// functional tests
		DEFINE_TEST_LONG_OPT(test_memcpy),
		DEFINE_TEST_LONG_OPT(test_saxpy_unified),
		DEFINE_TEST_LONG_OPT(test_host_async),
		DEFINE_TEST_LONG_OPT(test_host_async_batched),
		DEFINE_TEST_LONG_OPT(test_p2p),
		DEFINE_TEST_LONG_OPT(test_dev_sync),

		// performance tests - need to specify test explicitly to run
		DEFINE_TEST_LONG_OPT(test_p2p_bandwidth),
		DEFINE_TEST_LONG_OPT(test_p2p_bandwidth_p2p_off),
		DEFINE_TEST_LONG_OPT(test_p2p_bandwidth_kernel),

		// add new 'test_*' above

		{"perf", no_argument, 0, 'p'},
		{"yt_check", no_argument, 0, 'Y'},

		{"except", no_argument, 0, 'e'},
		{"gpu", required_argument, 0, 'g'},

		{"buf_size", required_argument, 0, 's'},
		{"threshold", required_argument, 0, 't'},

		{"completion_timeout", required_argument, 0, 'T'},
		{"kill_timeout", required_argument, 0, 'K'},

		{"show_gpu_count", no_argument, 0, 'c'},
		{"list_tests", no_argument, 0, 'l'},
		{"help",       no_argument, 0, 'h'},

		{0, 0, 0, 0}
	};

	int is_except = 0;
	int c = 0;
	while (1) {
		c = getopt_long(argc, argv, "pYeg:s:t:T:K:clh", long_options, NULL);
		if (c == -1)
			break;

		switch (c) {
		case 0:
			break;

		case 'p':
			__request_all_perf_tests();
			break;

		case 'Y':
			__request_all_func_tests();

			/* decide later which perf_tests to request as some GPU models are missing
			 * some features for perf_tests (e.g. V100 lacks nvlink, thus making p2p
			 * access for GPUs across numa nodes impossible), and parent running this
			 * function is free from any cuda* calls (so, no way to determine GPU model)
			 */
			// __request_all_perf_tests();

			*is_yt_check = 1;

			break;

		case 'e':
			is_except = 1;
			break;

		case 'g':
			// TODO: strtol() ?
			*gpu_idx = atoi(optarg);
			if (*gpu_idx < 0) {
				eprintf("Negative gpu index=%zd is given!\n", *gpu_idx);
				eprintf("\n");
				print_help(argv[0]);
				exit(EXIT_FAILURE);
			}
			break;

		case 's':
			err = parse_size(optarg, &buf_size);
			if (err != 0) {
				eprintf("'parse_size()' failed: err=%d\n", err);
				exit(EXIT_FAILURE);
			}
			if ((buf_size < 0) ||
			    ((size_t)buf_size < MIN_BUF_SIZE) ||
			    (buf_size % sizeof(ELEM_VALUE) != 0)) {
				eprintf("Bad buffer size: %zd (should be >= %zu, and multiple of %zu).\n", buf_size, MIN_BUF_SIZE, sizeof(ELEM_VALUE));
				eprintf("\n");
				print_help(argv[0]);
				exit(EXIT_FAILURE);
			}
			*elems_numb = buf_size / sizeof(ELEM_VALUE);
			printf("Using buffer size: %zd, elements number: %zu.\n", buf_size, *elems_numb);
			printf("\n");
			break;

		case 't':
			err = parse_size(optarg, threshold);
			if (err != 0) {
				eprintf("'parse_size()' failed: err=%d\n", err);
				exit(EXIT_FAILURE);
			}
			if ((*threshold < MIN_THRESHOLD) && (*threshold != NO_THRESHOLD)) {
				eprintf("Bad threshold value: %zd (should be >= %d, or %d for no threshold, default: %d).\n",
						*threshold, MIN_THRESHOLD, NO_THRESHOLD, DEFAULT_THRESHOLD);
				eprintf("\n");
				print_help(argv[0]);
				exit(EXIT_FAILURE);
			}

			if (*threshold == NO_THRESHOLD)
				printf("Using no threshold.\n");
			else
				printf("Using threshold: %zd.\n", *threshold);
			printf("\n");

			*is_manual_threshold = 1;

			break;

		case 'T':
			if (completion_timeout == NULL)
				break;

			err = parse_time(optarg, completion_timeout);
			if (err != 0) {
				eprintf("'parse_time()' failed: err=%d\n", err);
				exit(EXIT_FAILURE);
			}
			if (*completion_timeout < 0) {
				eprintf("Negative completion_timeout=%zd is given!\n", *completion_timeout);
				eprintf("\n");
				print_help(argv[0]);
				exit(EXIT_FAILURE);
			}

			break;

		case 'K':
			if (kill_timeout == NULL)
				break;

			err = parse_time(optarg, kill_timeout);
			if (err != 0) {
				eprintf("'parse_time()' failed: err=%d\n", err);
				exit(EXIT_FAILURE);
			}
			if (*kill_timeout < 0) {
				eprintf("Negative kill_timeout=%zd is given!\n", *kill_timeout);
				eprintf("\n");
				print_help(argv[0]);
				exit(EXIT_FAILURE);
			}

			break;

		case 'c':
			*show_gpu_count = 1;
			break;

		case 'l':
			print_tests_list();
			exit(EXIT_SUCCESS);
			break;

		case 'h':
			print_help(argv[0]);
			exit(EXIT_SUCCESS);
			break;

		case '?':
	        	break;

		default:
	        	eprintf("Unknown argument '%c': 'getopt_long()' returned character code 0%o\n", c, c);
	        	break;
	       }
	}

	if (optind < argc) {
		eprintf("non-option ARGV-elements: ");
		while (optind < argc)
			printf("%s ", argv[optind++]);
		printf("\n");
	}

	if (is_except)
		__invert_requested_tests_except_perf();

	size_t requested_tests_nr = __count_requested_tests();

	// if 'except' flag was not given and no tests were explicitly selected,
	// request all tests, as we came here because some other option (like '-g') was given
	if (!is_except && (requested_tests_nr == 0))
		requested_tests_nr = __request_all_non_perf_tests();

	return requested_tests_nr;
}

void print_help(const char *prog_name)
{
	printf("Usage: %s [OPTION...]\n", prog_name);
	printf("\n");
	printf("  -c, --show_gpu_count                         \t show detected gpus count and exit\n");
	printf("  -l, --list_tests                             \t list all available tests and exit\n");
	printf("  -h, --help                                   \t print this help and exit\n");
	printf("\n");
	printf("      without args                             \t run all non-perf tests on all gpus\n");
	printf("  -p, --perf                                   \t run all perf tests on all gpus\n");
	printf("      --test_name_1 --test_name_2 ...          \t run only selected tests\n");
	printf("  -e, --except --test_name_1 --test_name_2 ... \t run all but selected tests (doesn't run perf tests)\n");
	printf("  -Y, --yt_check                               \t run tests suitable for YT with GPU model autodetect and corresponding bandwidth threshold applied\n"
	       "                                               \t   (threshold provided for: A100-SXM4-40GB, A100-SXM-80GB)\n");
	// add new GPUs thresholds above
	printf("\n");
	printf("  -g, --gpu gpu_idx                            \t run tests only on selected gpu\n");
	printf("  -s, --buf_size s                             \t use array of size 's' bytes (where 's' >= %zu, and multiple of %zu, default: %llu),\n"
	       "                                               \t   supports [tT,gG,mM,kK,bB] size suffixes\n",
	                                                          MIN_BUF_SIZE, sizeof(ELEM_VALUE), DEFAULT_BUF_SIZE);
	printf("\n");
	printf("  -t, --threshold t                            \t use threshold of size 't' for perf tests (where 't' >= %d, %d - means no threshold, default: %d),\n"
	       "                                               \t   supports [tT,gG,mM,kK,bB] size suffixes\n", MIN_THRESHOLD, NO_THRESHOLD, DEFAULT_THRESHOLD);
	printf("  -T, --completion_timeout t                   \t wait at most 't' seconds for completion of cuda-check worker (default: %d),\n"
	       "                                               \t   supports [hH,mM,sS] time suffixes\n", DEFAULT_COMPLETION_TIMEOUT_SEC);
	printf("  -K, --kill_timeout t                         \t wait at most 't' seconds for cuda-check worker to die after sending kill to him after he exceeded completion timeout (default: %d),\n"
	       "                                               \t   supports [hH,mM,sS] time suffixes\n", DEFAULT_KILL_TIMEOUT_SEC);
}

void print_tests_list()
{
	printf("Following tests available:\n");
	printf("\n");

	printf("  Functional tests:\n");
	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++)
		if (!is_perf_test(i))
			printf("\t%s\n", tests_table[i].name);
	printf("\n");

	printf("  Performance tests (to run - specify test explicitly or pass '-p, --perf'):\n");
	for (size_t i = 0; i < TOTAL_TESTS_COUNT; i++)
		if (is_perf_test(i))
			printf("\t%s\n", tests_table[i].name);
}

int parse_size(const char *optarg, ssize_t *parsed_size)
{
	char *endptr = NULL;
	long long size = 0;

	errno = 0;
	size = strtoll(optarg, &endptr, 10);
	if ((errno == ERANGE && (size == LONG_LONG_MAX || size == LONG_LONG_MIN))
			|| (errno != 0 && size == 0)) {
		eprintf("'strtoll()' failed: %s\n", strerror(errno));
		return -1;
	}

	if (endptr == optarg) {
		eprintf("'strtoll()' failed: no digits were found\n");
		return -2;
	}

	switch (*endptr) {
	case 't':
	case 'T':
		size *= 1024;
		/* fallthrough */
	case 'g':
	case 'G':
		size *= 1024;
		/* fallthrough */
	case 'm':
	case 'M':
		size *= 1024;
		/* fallthrough */
	case 'k':
	case 'K':
		size *= 1024;
		switch (endptr[1]) {
		case 'b':
		case 'B':
		case '\0':
			break;
		default:
			goto err;
		}
	case 'b':
	case 'B':
	case '\0':
		break;
	default:
		goto err;
	}

	*parsed_size = (ssize_t)size;
	return 0;

err:
	eprintf("'parse_size()' failed: invalid size suffix: \"%s\"\n", endptr);
	return -3;
}

int parse_time(const char *optarg, ssize_t *parsed_time)
{
	char *endptr = NULL;
	long long time = 0;

	errno = 0;
	time = strtoll(optarg, &endptr, 10);
	if ((errno == ERANGE && (time == LONG_LONG_MAX || time == LONG_LONG_MIN))
			|| (errno != 0 && time == 0)) {
		eprintf("'strtoll()' failed: %s\n", strerror(errno));
		return -1;
	}

	if (endptr == optarg) {
		eprintf("'strtoll()' failed: no digits were found\n");
		return -2;
	}

	switch (*endptr) {
	case 'h':
	case 'H':
		time *= 60;
		/* fallthrough */
	case 'm':
	case 'M':
		time *= 60;
		/* fallthrough */
	case 's':
	case 'S':
	case '\0':
		break;
	default:
		goto err;
	}

	*parsed_time = (ssize_t)time;
	return 0;

err:
	eprintf("'parse_time()' failed: invalid time suffix: \"%s\"\n", endptr);
	return -3;
}
